import { _parseDocumentationTopics, formatTopicTitle } from '.';

describe('documentation-resolver', () => {
  describe(formatTopicTitle, () => {
    it('title cases the raw title', () => {
      expect(formatTopicTitle('testing')).toEqual('Testing');
      expect(formatTopicTitle('TESTING')).toEqual('Testing');
    });

    it('replaces dashes with spaces and upper cases each word', () => {
      expect(formatTopicTitle('testing-things')).toEqual('Testing Things');
      expect(formatTopicTitle('tesTing-otHer-thIngs')).toEqual(
        'Testing Other Things',
      );
    });
  });

  describe(_parseDocumentationTopics, () => {
    it('recursively creates topics for each folder level in a path and captures docAppPaths', () => {
      const { documents, rootTopic } = _parseDocumentationTopics(
        [
          'foo/bar/baz/README.md',
          'foo/bar/baz/info.md',
          'foo/bar/qux/README.md',
          'README.md',
          'info.md',
        ],
        [],
      );

      const document1 = {
        filePath: '/foo/bar/baz/README.md',
        title: formatTopicTitle('baz'),
        urlPath: '/foo/bar/baz/',
      };

      const document2 = {
        filePath: '/foo/bar/baz/info.md',
        title: formatTopicTitle('info'),
        urlPath: '/foo/bar/baz/info/',
      };

      const document3 = {
        filePath: '/foo/bar/qux/README.md',
        title: formatTopicTitle('qux'),
        urlPath: '/foo/bar/qux/',
      };

      const document4 = {
        filePath: '/README.md',
        title: formatTopicTitle('Root'),
        urlPath: '/',
      };

      const document5 = {
        filePath: '/info.md',
        title: formatTopicTitle('Info'),
        urlPath: '/info/',
      };

      expect(rootTopic).toEqual({
        document: document4,
        subTopics: {
          foo: {
            subTopics: {
              bar: {
                subTopics: {
                  baz: {
                    document: document1,
                    subTopics: {
                      info: {
                        document: document2,
                        subTopics: {},
                        title: 'Info',
                      },
                    },
                    title: 'Baz',
                  },
                  qux: {
                    document: document3,
                    subTopics: {},
                    title: 'Qux',
                  },
                },
                title: 'Bar',
              },
            },
            title: 'Foo',
          },
          info: {
            document: document5,
            subTopics: {},
            title: 'Info',
          },
        },
        title: 'Root',
      });

      expect(documents).toEqual([
        document1,
        document2,
        document3,
        document4,
        document5,
      ]);
    });

    it('does not include paths with ignored prefixes when parsing ', () => {
      const { documents, rootTopic } = _parseDocumentationTopics(
        ['leave_me', 'ignore_me_please'],
        ['ignore'],
      );

      expect(Object.keys(rootTopic.subTopics)).toEqual(['leave_me']);
      expect(documents).toEqual([]);
    });

    describe('treats README.md files as a root documentation file for a topic', () => {
      it('with root-level file', () => {
        const { documents, rootTopic } = _parseDocumentationTopics(
          ['README.md'],
          [],
        );

        const document = {
          filePath: '/README.md',
          title: formatTopicTitle('root'),
          urlPath: '/',
        };

        expect(rootTopic).toEqual({
          document,
          subTopics: {},
          title: 'Root',
        });
        expect(documents).toEqual([document]);
      });

      it('with nested README', () => {
        const {
          documents,
          rootTopic: { subTopics },
        } = _parseDocumentationTopics(['foo/README.md'], []);

        const document = {
          filePath: '/foo/README.md',
          title: formatTopicTitle('foo'),
          urlPath: '/foo/',
        };

        expect(subTopics).toEqual({
          foo: {
            document,
            subTopics: {},
            title: 'Foo',
          },
        });
        expect(documents).toEqual([document]);
      });
    });

    describe('generates a sub-topic for .md files that are not named "README.md"', () => {
      it('with root-level file', () => {
        const {
          documents,
          rootTopic: { subTopics },
        } = _parseDocumentationTopics(['TESTING.md'], []);

        const document = {
          filePath: '/TESTING.md',
          title: formatTopicTitle('TESTING'),
          urlPath: '/TESTING/',
        };

        expect(subTopics).toEqual({
          TESTING: {
            document,
            subTopics: {},
            title: 'Testing',
          },
        });
        expect(documents).toEqual([document]);
      });

      it('with nested file', () => {
        const {
          documents,
          rootTopic: { subTopics },
        } = _parseDocumentationTopics(['foo/TESTING.md'], []);

        const document = {
          filePath: '/foo/TESTING.md',
          title: formatTopicTitle('TESTING'),
          urlPath: '/foo/TESTING/',
        };

        expect(subTopics).toEqual({
          foo: {
            subTopics: {
              TESTING: {
                document,
                subTopics: {},
                title: 'Testing',
              },
            },
            title: 'Foo',
          },
        });
        expect(documents).toEqual([document]);
      });
    });
  });
});
