// ***********************************************
// Commands.ts is used to create various custom
// commands and overwrite existing commands.
//
// For more comprehensive examples of custom
// commands please read more here:
// https://on.cypress.io/custom-commands
// ***********************************************

import { Cypress, cy } from 'local-cypress';
import { addTacyhonCypressCommands } from 'tachyon-cypress';
import { exhaustedCase } from 'tachyon-utils';
import {
  KEY_NAV_THROTTLE_WAIT_MS,
  SCROLL_NAV_THROTTLE_WAIT_MS,
} from '../../src/config';

addTacyhonCypressCommands();

// safety buffer added after outwaiting tv-nav's throttling
const POST_THROTTLE_BUFFER_MS = 50;
// magnitude of scroll applied to synthetic wheel events
const WHEEL_DELTA_Y_PX = 100;

type TvNavDirection = 'down' | 'left' | 'right' | 'up';

type TvScrollDirection = 'down' | 'up';

declare global {
  // eslint-disable-next-line @typescript-eslint/no-namespace
  namespace Cypress {
    interface Chainable<Subject> {
      /**
       * Custom parent command for handling enter button functionality
       */
      tvEnter(): void;
      /**
       * Custom parent command which is used for navigation
       * @param {TvNavDirection}
       */
      tvNav(key: TvNavDirection): void;
      /**
       * Custom parent command for scrolling down page
       * @param {TvScrollDirection}
       */
      tvScroll(key: TvScrollDirection): void;
    }
  }
}

Cypress.Commands.add('tvNav', (key: TvNavDirection) => {
  cy.wait(KEY_NAV_THROTTLE_WAIT_MS + POST_THROTTLE_BUFFER_MS, { log: false });
  cy.get('body', { log: false }).type(`{${key}arrow}`, { log: false });

  Cypress.log({
    message: key,
    name: 'tvNav',
  });
});

Cypress.Commands.add('tvEnter', () => {
  // we silently handle the differences between input types
  cy.focused({ log: false }).then(($el) => {
    if ($el.is('input')) {
      cy.wrap($el, { log: false }).type('{enter}', { log: false });
    } else {
      cy.wrap($el, { log: false }).click({ log: false });
    }
  });
  Cypress.log({ name: 'tvEnter' });
});

Cypress.Commands.add('tvScroll', (key: TvScrollDirection) => {
  cy.wait(SCROLL_NAV_THROTTLE_WAIT_MS + POST_THROTTLE_BUFFER_MS, {
    log: false,
  });
  switch (key) {
    case 'down':
      cy.document({ log: false }).trigger('wheel', {
        deltaY: WHEEL_DELTA_Y_PX,
        log: false,
      });
      break;
    case 'up':
      cy.document({ log: false }).trigger('wheel', {
        deltaY: -WHEEL_DELTA_Y_PX,
        log: false,
      });
      break;
    default:
      exhaustedCase(key, null);
  }
  Cypress.log({ name: 'tvScroll' });
});

export {};
