/* eslint-disable import/order*/
const { ProvidePlugin } = require('webpack');
const SentryWebpackPlugin = require('@sentry/webpack-plugin');
const withBundleAnalyzer = require('@next/bundle-analyzer')({
  enabled: !!process.env.ANALYZE,
});
const withSourceMaps = require('@zeit/next-source-maps')({
  devtool: process.env.DOCKER_PRODUCTION ? 'hidden-source-map' : 'source-map',
});
const withWorkers = require('@zeit/next-workers');
const withFonts = require('next-fonts');
const es6Deps = require('tachyon-es6-deps-list');
const withTranspiledModules = require('next-transpile-modules')(es6Deps);
const { REDIRECT_TO_DESKTOP_WEB } =
  process.env.NODE_ENV === 'production'
    ? require('./dist/config')
    : require('./src/config');
const { buildId, nextAssetPrefix } = require('./env.config');

// Starshot next config
const NEXT_CONFIG = {
  webpack5: false,
  // ignore typescript errors in local dev
  typescript: {
    ignoreDevErrors: true,
  },

  // set the build ID to the git hash for determinism
  generateBuildId: () => buildId,

  workerLoaderOptions: {
    filename: `static/[name]-[hash].js`,
    publicPath: `${nextAssetPrefix}_next/`,
  },

  async headers() {
    return [
      {
        source: '/(.*)',
        headers: [
          {
            key: 'X-Frame-Options',
            value: 'deny',
          },
          {
            key: 'Content-Security-Policy',
            value: "frame-ancestors 'none'",
          },
        ],
      },
    ];
  },

  async redirects() {
    return [
      // https://github.com/vercel/next.js/issues/17189
      {
        source: '/_error',
        destination: '/not-found',
        permanent: false,
      },
      ...REDIRECT_TO_DESKTOP_WEB.map((source) => ({
        source,
        destination: `https://www.twitch.tv${source}`,
        permanent: false,
      })),
    ];
  },

  // custom webpack config
  webpack(config, { defaultLoaders, dev, isServer }) {
    // when troubleshooting Chrome 38 transpilation issues, uncomment this section
    // if (!isServer) {
    //   config.optimization.minimize = false;
    //   config.optimization.minimizer = undefined;
    // }

    config.module.rules.push({
      test: /\.css$/,
      use: 'raw-loader',
    });

    // set up aliases
    const alias = config.resolve.alias || {};

    // Next 9.5 supports `next build --profile` but our current build process
    // is too complex to leverage that
    if (process.env.REACT_PROFILE) {
      alias['react-dom$'] = 'react-dom/profiling';
      alias['scheduler/tracing$'] = 'scheduler/tracing-profiling';
    }

    config.resolve.alias = {
      ...alias,
      // the focus-visible polyfill is injected by core-ui to help keyboard a11y
      // but starshot manages all of its own focus states and is explicitly
      // keyboard/d-pad compatible so this just gets in the way
      'focus-visible': 'tachyon-noop',
      // react-router isn't used by Tachyon apps but is required by core-ui
      'react-router': 'tachyon-noop',
      'react-router-dom': 'tachyon-noop',
    };

    // ensure that bundle order doesn't result in a regeneratorRuntime error
    config.plugins.push(
      new ProvidePlugin({
        regeneratorRuntime: 'regenerator-runtime/runtime',
      }),
    );

    if (!isServer) {
      config.plugins.push(
        new SentryWebpackPlugin({
          dryRun: !process.env.SENTRY_AUTH_TOKEN,
          include: '.next',
          release: buildId,
          urlPrefix: '~/_next',
        }),
      );
    }

    return config;
  },
};

module.exports = (phase, { defaultConfig }) => {
  const baseConfig = withSourceMaps(
    withTranspiledModules(withWorkers(withFonts(NEXT_CONFIG))),
  );

  if (process.env.DOCKER_PRODUCTION) {
    return baseConfig;
  }

  return withBundleAnalyzer(baseConfig);
};
