import { stringify } from 'querystring';
import { graphql } from 'react-relay/hooks';
import { convertToUnsafeID } from 'tachyon-relay';
import type { OmitRefType } from 'tachyon-type-library';
import { ITEM_TRACKING_ID } from '../../../config';
import type {
  Language,
  mediaFeed_QueryResponse,
} from '../__generated__/mediaFeed_Query.graphql';
import type { PreviewImageSet } from '../utils';
import { getPreviewImageSet } from '../utils';
import type { getLiveStreams_streamEdge } from './__generated__/getLiveStreams_streamEdge.graphql';

type Stream = OmitRefType<getLiveStreams_streamEdge>;

export type LiveStream = {
  broadcastLanguage: Language;
  broadcasterId: string;
  broadcasterName: string;
  categoryId: string;
  categoryName: string;
  previewImageSet: PreviewImageSet;
  title: string;
  twitchStreamURL: URL;
  viewersCount: number;
};

// Type used for checking non-nullability of streams
type CompleteStream = {
  node: {
    broadcastLanguage: Language;
    broadcaster: {
      broadcastSettings: {
        game: {
          displayName: string;
          id: string;
        };
        title: string;
      };
      displayName: string;
      id: string;
      login: string;
    };
    createdAt: string;
    id: string;
    previewImageURL: string;
    viewersCount: number;
  };
  trackingID: string;
};

export function getLiveStreams(
  { recommendedStreams }: mediaFeed_QueryResponse,
  domain: string,
): Array<LiveStream> | undefined {
  return recommendedStreams?.edges?.reduce<Array<LiveStream>>(
    (liveStreams, edge) => {
      const liveStream = getFormattedLiveStream(edge, domain);
      if (liveStream) {
        liveStreams.push(liveStream);
      }
      return liveStreams;
    },
    [],
  );
}

export function isCompleteLiveStream(
  item: Stream | null,
): item is CompleteStream {
  return (
    !!item?.node?.broadcaster?.broadcastSettings?.game &&
    item.node.previewImageURL !== null &&
    item.node.viewersCount !== null &&
    item.node.broadcastLanguage !== null
  );
}

// istanbul ignore next: trivial
function getFormattedLiveStream(
  item: Stream | null,
  domain: string,
): LiveStream | undefined {
  if (isCompleteLiveStream(item)) {
    const stream = item.node;
    const broadcaster = stream.broadcaster;
    const trackingParams = stringify({
      [ITEM_TRACKING_ID]: item.trackingID,
    });

    const channelLink = new URL(
      `/${broadcaster.login}?${trackingParams}`,
      domain,
    );
    return {
      broadcastLanguage: stream.broadcastLanguage,
      broadcasterId: convertToUnsafeID(broadcaster.id),
      broadcasterName: broadcaster.displayName,
      categoryId: convertToUnsafeID(broadcaster.broadcastSettings.game.id),
      categoryName: broadcaster.broadcastSettings.game.displayName,
      previewImageSet: getPreviewImageSet(item.node.previewImageURL),
      title: broadcaster.broadcastSettings.title,
      twitchStreamURL: channelLink,
      viewersCount: stream.viewersCount,
    };
  }
}

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment getLiveStreams_streamEdge on RecommendedStreamsEdge {
    trackingID
    node {
      broadcaster {
        displayName
        id
        login
        broadcastSettings {
          game {
            displayName
            id
          }
          title
        }
      }
      broadcastLanguage
      createdAt
      id
      previewImageURL
      viewersCount
    }
  }
`;
