import { datatype } from 'faker';
import { getLiveStreams, isCompleteLiveStream } from '.';

describe('getLiveStreams', () => {
  describe('returns undefined', () => {
    it('when recommendedStreams is null', () => {
      expect(getLiveStreams({ recommendedStreams: null }, '')).toBeUndefined();
    });

    it('when edges is null', () => {
      expect(
        getLiveStreams({ recommendedStreams: { edges: null } }, ''),
      ).toBeUndefined();
    });
  });

  describe('returns an empty array', () => {
    it('when there are no edges provided', () => {
      expect(
        getLiveStreams({ recommendedStreams: { edges: [] } }, ''),
      ).toHaveLength(0);
    });

    it('when none of the edges are complete streams', () => {
      const streams = getLiveStreams(
        {
          recommendedStreams: {
            edges: [
              { node: null, trackingID: datatype.uuid() },
              { node: null, trackingID: datatype.uuid() },
            ],
          },
        },
        'https://tv.twitch.tv',
      );
      expect(streams).toHaveLength(0);
    });
  });
});

const completeStream = {
  broadcaster: { broadcastSettings: { game: true } },
  language: 'EN',
  previewImageURL: 'test',
  viewersCount: datatype.number(),
};
describe('isCompleteLiveStream', () => {
  it.each`
    item                                                                                   | itemIsCompleteStream | condition
    ${{ node: completeStream }}                                                            | ${true}              | ${'complete stream'}
    ${{ node: { ...completeStream, previewImageURL: null } }}                              | ${false}             | ${'previewImageURL is null'}
    ${{ node: { ...completeStream, viewersCount: null } }}                                 | ${false}             | ${'viewersCount is null'}
    ${{ node: { ...completeStream, broadcaster: null } }}                                  | ${false}             | ${'broadcaster is null'}
    ${{ node: { ...completeStream, broadcaster: { broadcastSettings: null } } }}           | ${false}             | ${'broadcastSettings is null'}
    ${{ node: { ...completeStream, broadcaster: { broadcastSettings: { game: null } } } }} | ${false}             | ${'game is null'}
    ${{ node: null }}                                                                      | ${false}             | ${'node is null'}
    ${null}                                                                                | ${false}             | ${'item is null'}
  `(
    'returns $itemIsCompleteStream when $condition ',
    ({ item, itemIsCompleteStream }) => {
      expect(isCompleteLiveStream(item)).toBe(itemIsCompleteStream);
    },
  );
});
