import Bowser from 'bowser';
import { graphql } from 'react-relay/hooks';
import type { ErrorLogPayload } from 'tachyon-logger';
import { logger } from 'tachyon-logger';
import type { TachyonApiHandler } from 'tachyon-next-types';
import { HTTPStatusCode } from 'tachyon-type-library';
import { getCurrentTwitchDomain, uniqueIDGenerator } from 'tachyon-utils';
import type { StarshotRequestExtensions } from '../../config';
import { CLIENT_APP } from '../../config';
import { apiRelayQuery } from '../utils';
import type { mediaFeed_QueryResponse } from './__generated__/mediaFeed_Query.graphql';
import type { LiveStream } from './getLiveStreams';
import { getLiveStreams } from './getLiveStreams';
import type { ApiJsonErrorResponse, Provider } from './utils';
import {
  ERROR_BAD_PARAM_CATEGORY_IDS,
  ERROR_BAD_PARAM_FIRST,
  ERROR_DOMAIN_DOES_NOT_EXIST,
  ERROR_GQL_FAILED_FETCH,
  ERROR_MISSING_USER_AGENT,
  ERROR_PARTIAL_SUCCESS,
  ERROR_RESULT_IS_UNDEFINED,
  ERROR_STREAMS_DONT_EXIST,
  ERROR_STREAMS_IS_EMPTY,
  convertQueryParamToWholeNumber,
  getProvider,
} from './utils';

const REQUEST_ID_LENGTH = 16;
const DEFAULT_NUMBER_OF_RESULTS = 20;

type MediaFeed = {
  items: {
    liveStreams: Array<LiveStream>;
  };
  provider: Provider;
};

export const mediaFeed: TachyonApiHandler<
  ApiJsonErrorResponse | MediaFeed,
  StarshotRequestExtensions
> = async (req, res) => {
  // Check if we have a valid user-agent for security
  if (
    !req.headers['user-agent'] ||
    !Bowser.parse(req.headers['user-agent']).browser.name
  ) {
    logger.error({
      category: 'MediaFeed API Response',
      context: { headers: req.headers.toString() },
      message: 'Request with no user-agent rejected',
      package: 'starshot-app',
    });
    res.status(HTTPStatusCode.BadRequest).json(ERROR_MISSING_USER_AGENT);
    return;
  }

  // Check if the requested domain is a twitch domain
  if (!getCurrentTwitchDomain(req.hostname)) {
    res.status(HTTPStatusCode.BadRequest).json(ERROR_DOMAIN_DOES_NOT_EXIST);
    return;
  }
  const domain = `https://${req.hostname}`;

  const {
    query: { 'category-id': categoryIds, first },
  } = req;

  // If `first` is supplied, check if it's a valid input
  const convertedFirst = convertQueryParamToWholeNumber(first);
  if ((first && !convertedFirst) || (convertedFirst && convertedFirst > 100)) {
    res.status(HTTPStatusCode.BadRequest).json(ERROR_BAD_PARAM_FIRST);
    return;
  }

  // If category-id is supplied, check if it's valid
  if (categoryIds !== undefined) {
    const categories = Array.isArray(categoryIds) ? categoryIds : [categoryIds];
    const hasBadCategoryId = categories.find(
      (category) => !category || !convertQueryParamToWholeNumber(category),
    );
    if (hasBadCategoryId) {
      res.status(HTTPStatusCode.BadRequest).json(ERROR_BAD_PARAM_CATEGORY_IDS);
      return;
    }
  }

  const relayQueryVariables = {
    context: {
      clientApp: CLIENT_APP,
      platform: req.tachyon.platform,
    },
    filters: {
      categoryIDs: categoryIds,
    },
    first: convertedFirst ?? DEFAULT_NUMBER_OF_RESULTS,
    location: 'TV_APPS',
    recRequestID: uniqueIDGenerator(REQUEST_ID_LENGTH),
  };

  let queryResults;
  try {
    queryResults = await apiRelayQuery(
      req,
      res,
      mediaFeedQuery,
      relayQueryVariables,
    );
  } catch (error) {
    logger.error(error as ErrorLogPayload);
    res.status(HTTPStatusCode.InternalServerError).json(ERROR_GQL_FAILED_FETCH);
    return;
  }

  // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
  if (!queryResults) {
    res
      .status(HTTPStatusCode.InternalServerError)
      .json(ERROR_RESULT_IS_UNDEFINED);
    return;
  }

  // If we have a partial response we'll return a 500. We'll consider partial success
  // more for future versions
  if (queryResults.isGQLPartialSuccess) {
    res.status(HTTPStatusCode.InternalServerError).json(ERROR_PARTIAL_SUCCESS);
    return;
  }

  if (!isMediaFeed_QueryResponse(queryResults.relayQueryResponse)) {
    res
      .status(HTTPStatusCode.InternalServerError)
      .json(ERROR_STREAMS_DONT_EXIST);
    return;
  }

  const liveStreams = getLiveStreams(queryResults.relayQueryResponse, domain);

  if (!liveStreams?.length) {
    res.status(HTTPStatusCode.InternalServerError).json(ERROR_STREAMS_IS_EMPTY);
    return;
  }

  res.json({
    items: {
      liveStreams,
    },
    provider: getProvider(domain),
  });
};

// istanbul ignore next: trivial
function isMediaFeed_QueryResponse(
  queryResponse: mediaFeed_QueryResponse | any,
): queryResponse is mediaFeed_QueryResponse {
  return !!(queryResponse as mediaFeed_QueryResponse).recommendedStreams;
}

const mediaFeedQuery = graphql`
  query mediaFeed_Query(
    $recRequestID: ID!
    $location: String!
    $context: RecommendationsContext!
    $first: Int
    $filters: StreamRecommendationsFilters
  ) {
    recommendedStreams(
      recRequestID: $recRequestID
      location: $location
      context: $context
      first: $first
      filters: $filters
    ) {
      edges {
        ...getLiveStreams_streamEdge @relay(mask: false)
      }
    }
  }
`;
