import { MockUserAgent, Platform } from 'tachyon-environment';
import type { TachyonApiRequest } from 'tachyon-next-types';
import { HTTPStatusCode } from 'tachyon-type-library';
import type { StarshotRequestExtensions } from '../../config';
import { apiRelayQuery } from '../utils';
import { getLiveStreams } from './getLiveStreams';
import {
  ERROR_BAD_PARAM_CATEGORY_IDS,
  ERROR_BAD_PARAM_FIRST,
  ERROR_DOMAIN_DOES_NOT_EXIST,
  ERROR_GQL_FAILED_FETCH,
  ERROR_MISSING_USER_AGENT,
  ERROR_PARTIAL_SUCCESS,
  ERROR_RESULT_IS_UNDEFINED,
  ERROR_STREAMS_DONT_EXIST,
  ERROR_STREAMS_IS_EMPTY,
} from './utils';
import { mediaFeed } from '.';

jest.mock('tachyon-logger', () => ({
  ...jest.requireActual('tachyon-logger'),
  logger: {
    error: jest.fn(),
  },
}));

jest.mock('../utils');
const mockApiRelayQuery = apiRelayQuery as jest.Mock;

jest.mock('./getLiveStreams');
const mockGetLiveStreams = getLiveStreams as jest.Mock;

let res: any;
let mockJson: any;
let mockStatus: any;
let req: any;

function setupMocks() {
  mockJson = jest.fn();
  mockStatus = jest.fn();

  res = {
    json: mockJson,
    status: mockStatus,
  };
  mockJson.mockImplementation(() => res);
  mockStatus.mockImplementation(() => res);

  req = {
    headers: {
      'user-agent': MockUserAgent.macOS.Chrome,
    },
    hostname: 'tv.twitch.tv',
    query: {
      'category-id': ['123'],
      first: '20',
    },
    tachyon: {
      platform: Platform.StarshotDev,
    },
  };
}
describe(mediaFeed, () => {
  beforeEach(() => {
    jest.resetAllMocks();
    setupMocks();
  });

  it('errors when first is greater than 100', async () => {
    req.query.first = '200';
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_BAD_PARAM_FIRST);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('errors when first is less than 1', async () => {
    req.query.first = '0';
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_BAD_PARAM_FIRST);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('errors when first is a word', async () => {
    req.query.first = 'test';
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_BAD_PARAM_FIRST);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('errors when categoryIds contains a non-whole number value', async () => {
    req.query['category-id'] = ['123', 'test'];
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_BAD_PARAM_CATEGORY_IDS);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('errors when categoryIds is a non-whole number', async () => {
    req.query['category-id'] = 'test';
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_BAD_PARAM_CATEGORY_IDS);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('does not error when categoryIds is a whole number', async () => {
    req.query['category-id'] = '123';
    mockApiRelayQuery.mockReturnValueOnce({
      isGQLPartialSuccess: false,
      relayQueryResponse: { recommendedStreams: true },
    });
    mockGetLiveStreams.mockReturnValueOnce(['one', 'two']);
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockStatus).not.toHaveBeenCalled();
  });

  it('errors when user-agent is missing from headers', async () => {
    req.headers['user-agent'] = '';
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_MISSING_USER_AGENT);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('erros when hostname is undefined', async () => {
    req.hostname = false;
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_DOMAIN_DOES_NOT_EXIST);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.BadRequest);
  });

  it('errors when fetch throws', async () => {
    mockApiRelayQuery.mockRejectedValueOnce({});
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_GQL_FAILED_FETCH);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.InternalServerError);
  });

  it('errors when gql reults are falsey', async () => {
    mockApiRelayQuery.mockReturnValueOnce(false);
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_RESULT_IS_UNDEFINED);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.InternalServerError);
  });

  it('errors when gql indicates partial success', async () => {
    mockApiRelayQuery.mockReturnValueOnce({ isGQLPartialSuccess: true });
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_PARTIAL_SUCCESS);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.InternalServerError);
  });
  it('errors when recommendedStreams is undefined', async () => {
    mockApiRelayQuery.mockReturnValueOnce({
      isGQLPartialSuccess: false,
      relayQueryResponse: { recommendedStreams: undefined },
    });
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_STREAMS_DONT_EXIST);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.InternalServerError);
  });

  it('errors when getLiveStreams returns an empty list', async () => {
    mockApiRelayQuery.mockReturnValueOnce({
      isGQLPartialSuccess: false,
      relayQueryResponse: { recommendedStreams: true },
    });
    mockGetLiveStreams.mockReturnValueOnce([]);
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockJson).toHaveBeenCalledWith(ERROR_STREAMS_IS_EMPTY);
    expect(mockStatus).toHaveBeenCalledWith(HTTPStatusCode.InternalServerError);
  });

  it('does not error when we have a happy path', async () => {
    mockApiRelayQuery.mockReturnValueOnce({
      isGQLPartialSuccess: false,
      relayQueryResponse: { recommendedStreams: true },
    });
    mockGetLiveStreams.mockReturnValueOnce(['one', 'two']);
    await mediaFeed(req as TachyonApiRequest<StarshotRequestExtensions>, res);

    expect(mockStatus).not.toHaveBeenCalled();
  });
});
