import type { HeaderOrParamValue } from 'tachyon-utils';
import { flattenHeaderOrParam, memoizeSimpleAll } from 'tachyon-utils';

type TwitchAsset = {
  png: URL;
  svg: URL;
};
export type Provider = {
  logo: TwitchAsset;
  name: 'Twitch';
  wordmark: TwitchAsset;
};

// istanbul ignore next: trivial
export const getProvider = memoizeSimpleAll(
  (domain: string): Provider => ({
    logo: {
      png: new URL('/static/starshot/images/TwitchGlitchPurple.png', domain),
      svg: new URL('/static/starshot/images/TwitchGlitchPurple.svg', domain),
    },
    name: 'Twitch',
    wordmark: {
      png: new URL(
        '/static/starshot/images/TwitchExtrudedWordmarkPurple.png',
        domain,
      ),
      svg: new URL(
        '/static/starshot/images/TwitchExtrudedWordmarkPurple.svg',
        domain,
      ),
    },
  }),
);

export type PreviewImageSet = {
  large: string;
  medium: string;
  small: string;
};

// istanbul ignore next: trivial
export function getPreviewImageSet(templateString: string): PreviewImageSet {
  return {
    large: getScaledPreviewImageURL(templateString, 1),
    medium: getScaledPreviewImageURL(templateString, 0.5),
    small: getScaledPreviewImageURL(templateString, 0.2),
  };
}

function getScaledPreviewImageURL(
  templateString: string,
  multiplier: number,
): string {
  return templateString
    .replace('{width}', `${1920 * multiplier}`)
    .replace('{height}', `${1080 * multiplier}`);
}

export function convertQueryParamToWholeNumber(
  param: HeaderOrParamValue,
): number | null {
  const parsedParam = parseInt(flattenHeaderOrParam(param) ?? '');
  if (
    isNaN(parsedParam) ||
    parsedParam < 1 ||
    parsedParam.toString() !== param
  ) {
    return null;
  }

  return parsedParam;
}

export type ApiJsonErrorResponse = {
  error: string;
};

export const ERROR_PARTIAL_SUCCESS: ApiJsonErrorResponse = {
  error: 'Result is only a partial success',
};
export const ERROR_STREAMS_DONT_EXIST: ApiJsonErrorResponse = {
  error: 'Recommended streams do not exist',
};
export const ERROR_STREAMS_IS_EMPTY: ApiJsonErrorResponse = {
  error: 'No recommended streams available',
};
export const ERROR_RESULT_IS_UNDEFINED: ApiJsonErrorResponse = {
  error: 'Query results are undefined',
};
export const ERROR_GQL_FAILED_FETCH: ApiJsonErrorResponse = {
  error: 'Can not fetch data from GQL',
};
export const ERROR_DOMAIN_DOES_NOT_EXIST: ApiJsonErrorResponse = {
  error: 'The requested environment is not supported',
};
export const ERROR_MISSING_USER_AGENT: ApiJsonErrorResponse = {
  error: 'Bad request',
};
export const ERROR_BAD_PARAM_FIRST: ApiJsonErrorResponse = {
  error: 'Bad request: `first` must be a whole number between 0 and 101',
};

export const ERROR_BAD_PARAM_CATEGORY_IDS: ApiJsonErrorResponse = {
  error: 'Bad request: `category-id` must be a whole number greater than 0',
};
