import type {
  GraphQLTaggedNode,
  Variables as RelayQueryVariables,
} from 'react-relay/hooks';
import { fetchQuery } from 'react-relay/hooks';
import { getCurrentUserOnServer, logoutOnServer } from 'tachyon-auth-server';
import { logger } from 'tachyon-logger';
import type { TachyonApiRequest, TachyonApiResponse } from 'tachyon-next-types';
import { initEnvironment } from 'tachyon-relay';
import {
  createDeviceIDOnServer,
  getDeviceIDOnServer,
} from 'tachyon-server-utils';
import type { StarshotRequestExtensions } from '../../config';

/**
 * utility function for server-side relay queries. This will accept any
 * relay query and an object containing query variables and returns a
 * promise containing a bool indicating if we had a partial success and
 * the query response itself.
 *
 * @param {TachyonApiRequest<StarshotRequestExtensions>} req - req provided by Next
 * @param {TachyonApiResponse} res - res provided by Next
 * @param {GraphQLTaggedNode} relayQuery - reference to a GQL query
 * @param {Variables} relayQueryVariables - object containing the args to pass to the query
 */
// istanbul ignore next: low value
export async function apiRelayQuery(
  req: TachyonApiRequest<StarshotRequestExtensions>,
  res: TachyonApiResponse,
  relayQuery: GraphQLTaggedNode,
  relayQueryVariables: RelayQueryVariables,
): Promise<{ isGQLPartialSuccess: boolean; relayQueryResponse: unknown }> {
  let isGQLPartialSuccess = false;
  const { authorizationToken } = getCurrentUserOnServer(req, res);
  let deviceId = getDeviceIDOnServer(req, res);
  if (deviceId === undefined) {
    logger.error({
      category: 'API GQL Query',
      context: { headers: req.headers.toString() },
      message: 'Request with no device ID sent',
      package: 'starshot-app',
    });
    deviceId = createDeviceIDOnServer(res);
  }
  const relayEnvironment = initEnvironment({
    fetchQueryOpts: {
      authorization: {
        token: authorizationToken,
        unauthorizedHandler: () => {
          logoutOnServer(res);
        },
      },
      deviceId,
      language: req.tachyon.intlData.preferredLanguageTags[0],
      nonFatalErrorsHandler: () => {
        isGQLPartialSuccess = true;
      },
    },
  });

  const relayQueryResponse = await fetchQuery(
    relayEnvironment,
    relayQuery,
    relayQueryVariables,
  ).toPromise();

  return { isGQLPartialSuccess, relayQueryResponse };
}
