import type { FC } from 'react';
import type { ChatWorkerInstance } from 'tachyon-chat';
import type { ChatRootProps } from 'tachyon-chat-ui';
import { ChatPane, ChatRoot } from 'tachyon-chat-ui';
import { useIntl } from 'tachyon-intl';
import { useCustomLatency } from 'tachyon-latency-tracker';
import { convertToUnsafeID } from 'tachyon-relay';
import { useConstCallback } from 'tachyon-utils';
import {
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Overflow,
  SVGAsset,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../routing';
import { FocusableTextButton } from '../common';
// @ts-expect-error: WebWorkers are necessarily (untyped) JS
import ChatWorker from './chat.worker';

// Since we do not allow scrolling through chat history, the buffer should be
// set as small as possible such that messages barely overflow the page.
const CHAT_BUFFER_SIZE = 30;

// Twilight has the following ramp, so we preserve those ratios:
// - font-size @ 12px (1rem)
// - badges @ 18px (native image size)
// - emotes @ 28px (native image size)
const CHAT_BADGE_SIZE = '1.5rem';
const CHAT_EMOTE_SIZE = '2.4rem';

const INTERACTION_CONTENT_JOIN_CONVERSATION = 'join_conversation';

type ChatProps = Pick<ChatRootProps, 'channel' | 'clientApiId'> & {
  focusIndex: number;
};

export const Chat: FC<ChatProps> = ({ channel, clientApiId, focusIndex }) => {
  const { formatMessage } = useIntl();
  const reportLatencyEvent = useCustomLatency();
  const reportChatConnected = useConstCallback(
    reportLatencyEvent('chatConnected'),
  );

  return (
    <Layout
      display={Display.Flex}
      flexDirection={FlexDirection.Column}
      fullHeight
    >
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        flexGrow={1}
        justifyContent={JustifyContent.End}
        overflow={Overflow.Hidden}
      >
        <ChatRoot
          bufferSize={CHAT_BUFFER_SIZE}
          channel={{ id: convertToUnsafeID(channel.id), login: channel.login }}
          clientApiId={clientApiId}
          getWorker={() => {
            // eslint-disable-next-line @typescript-eslint/no-unsafe-call
            return new ChatWorker() as ChatWorkerInstance;
          }}
        >
          <ChatPane
            onFirstMessage={reportChatConnected}
            panePadding={{ top: 2 }}
            sizes={{ badgeSize: CHAT_BADGE_SIZE, emoteSize: CHAT_EMOTE_SIZE }}
          />
        </ChatRoot>
      </Layout>
      <Layout padding={{ x: 2, y: 1 }}>
        <FocusableTextButton
          focusIndex={focusIndex}
          fullWidth
          icon={SVGAsset.Mobile}
          linkTo="/deferToRenderLink"
          renderLink={renderTachyonLink({
            interactionContent: INTERACTION_CONTENT_JOIN_CONVERSATION,
            route: RouteName.JoinConversation,
            routeParams: { login: channel.login },
          })}
          withUnfocusBackground
        >
          {formatMessage('Join the conversation', 'JoinConversation')}
        </FocusableTextButton>
      </Layout>
    </Layout>
  );
};

Chat.displayName = 'Chat';
