import type { FC } from 'react';
import { createContext, useContext, useState } from 'react';
import { isBrowser, useEffectOnce } from 'tachyon-utils';

function isVisible() {
  if (!isBrowser()) {
    return true;
  }
  return !document.hidden;
}

export const visibleContext = createContext(true);

export const AppVisibilityRoot: FC = ({ children }) => {
  const [visible, setVisible] = useState(isVisible);

  // http://webostv.developer.lge.com/develop/app-developer-guide/web-app-lifecycle/
  // https://developer.tizen.org/development/articles/application-visibility?langredirect=1
  // Browser Support:
  // https://caniuse.com/#feat=pagevisibility
  // Note: In Chrome 38 the visibilitychange event is fired twice
  // Chrome Bug: https://bugs.chromium.org/p/chromium/issues/detail?id=422163
  useEffectOnce(() => {
    const visibilityHandler = () => {
      setVisible(isVisible);
    };

    document.addEventListener('visibilitychange', visibilityHandler);

    return () => {
      document.removeEventListener('visibilitychange', visibilityHandler);
    };
  });

  return <visibleContext.Provider children={children} value={visible} />;
};

AppVisibilityRoot.displayName = 'AppVisibilityRoot';

export function useAppVisibility(): boolean {
  return useContext(visibleContext);
}
