import type { FC } from 'react';
import { useDebouncedState } from 'tachyon-utils';
import { AlignItems, Display, Layout } from 'twitch-core-ui';
import type { RequireExactlyOne } from 'type-fest';
import { CARD_FOCUS_META_DEBOUNCE_MS } from '../../../../config';
import type { BannerBackgroundProps } from '../BannerBackground';
import { BannerBackground } from '../BannerBackground';
import type { BannerCategoryImageProps } from '../BannerCategoryImage';
import { BannerCategoryImage } from '../BannerCategoryImage';

export type BannerData<BannerContentData> = {
  /**
   * Background image src.
   */
  backgroundImageSrc: BannerBackgroundProps['backgroundImageSrc'];
  /**
   * Category image src. Must be used with `showCategoryImage` prop.
   */
  categoryImageSrc?: BannerCategoryImageProps['boxArtURL'];
  /**
   * Data passed to the `<BannerContent>` when the banner is rendered/updated.
   */
  contentData?: BannerContentData;
};

export type BannerDataUpdateFn<T> = (bannerData: BannerData<T>) => void;

export type UpdateBannerData<T> = {
  (bannerData: BannerData<T>): void;
  __getInitialData: () => BannerData<T>;
  __setRef: (updater: BannerDataUpdateFn<T>) => void;
};

export type BannerContentBaseProps<BannerContentData> = {
  data: BannerContentData | undefined;
};

export type BannerProps<
  BannerContentData,
  BannerContentProps extends BannerContentBaseProps<BannerContentData>,
> = RequireExactlyOne<
  {
    /**
     * Function component for populating content overlaid on the banner (other
     * than a category image which is handled by the component directly). It
     * receives the current `contentData` value as a `data` prop. This can be
     * omitted if you only want the background image, as it will protect its own
     * height.
     */
    BannerContent?: FC<BannerContentProps>;
    /**
     * Static props to pass to `<BannerContent>` when rendering. Must be stable
     * (probably memoized) to avoid perf regressions.
     */
    bannerContentProps?: Omit<BannerContentProps, 'data'>;
    /**
     * Controls whether a category image is rendered, with appropriate layout
     * considerations for the other banner content.
     */
    showCategoryImage?: boolean;
    /**
     * Banner data for use when the banner does not change. Mutually exclusive
     * with `updateBannerData`.
     */
    staticBannerData: BannerData<BannerContentData>;
    /**
     * Update function (generated by `useBannerData`) for communicating dynamic
     * banner data. Mutually exclusive with `staticBannerData`.
     */
    updateBannerData: UpdateBannerData<BannerContentData>;
  },
  'staticBannerData' | 'updateBannerData'
>;

/**
 * The actual Banner UI and state container. Should not be used outside of the
 * `BannerPage` to ensure proper layout on Chrome 38.
 */
export function Banner<
  BannerContentData,
  BannerContentProps extends BannerContentBaseProps<BannerContentData> = BannerContentBaseProps<BannerContentData>,
>({
  BannerContent,
  bannerContentProps,
  showCategoryImage,
  staticBannerData,
  updateBannerData,
}: BannerProps<BannerContentData, BannerContentProps>): JSX.Element {
  const [bannerData, setBannerData] = useDebouncedState<
    // bannerData should always be defined but RequireExactlyOne can
    // confuse TS so we proceed safely
    BannerData<BannerContentData> | undefined
  >(
    updateBannerData ? updateBannerData.__getInitialData() : staticBannerData,
    CARD_FOCUS_META_DEBOUNCE_MS,
    {
      trailing: true,
    },
  );
  updateBannerData?.__setRef(setBannerData);

  return (
    <BannerBackground backgroundImageSrc={bannerData?.backgroundImageSrc}>
      <Layout alignItems={AlignItems.Center} display={Display.Flex} fullHeight>
        {showCategoryImage && (
          <BannerCategoryImage boxArtURL={bannerData?.categoryImageSrc} />
        )}
        <Layout padding={showCategoryImage ? 3 : 0}>
          {BannerContent && (
            <BannerContent
              {...(bannerContentProps as BannerContentProps)}
              data={bannerData?.contentData}
            />
          )}
        </Layout>
      </Layout>
    </BannerBackground>
  );
}

Banner.displayName = 'Banner';
