import { internet } from 'faker';
import type { FC } from 'react';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { BannerBackground } from '../BannerBackground';
import { BannerCategoryImage } from '../BannerCategoryImage';
import type { BannerData } from '.';
import { Banner } from '.';

const mockSetState = jest.fn();
jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  useDebouncedState: jest.fn((data: any) => [data, mockSetState]),
}));

const MockComponent: FC<any> = ({ foo }) => (
  <div className="bannerContent">{foo}</div>
);

describe(Banner, () => {
  const contentData = 'content data';
  const bannerData: BannerData<typeof contentData> = {
    backgroundImageSrc: internet.url(),
    categoryImageSrc: internet.url(),
    contentData,
  };

  const setup = createShallowWrapperFactory(Banner, () => ({
    BannerContent: MockComponent,
    bannerContentProps: { foo: 'bar' },
    bannerData: undefined,
    showCategoryImage: true,
    updateBannerData: {
      __getInitialData: jest.fn(() => bannerData),
      __setRef: jest.fn(),
    } as any,
  }));

  it('renders using the updateBannerData prop for getting data', () => {
    const { props, wrapper } = setup();

    expect(wrapper.find(BannerBackground)).toHaveProp({
      backgroundImageSrc: bannerData.backgroundImageSrc,
    });
    expect(wrapper.find(BannerCategoryImage)).toHaveProp({
      boxArtURL: bannerData.categoryImageSrc,
    });
    expect(wrapper.find('Layout[padding=3]')).toExist();

    expect(props.updateBannerData!.__getInitialData).toHaveBeenCalled();
    expect(props.updateBannerData!.__setRef).toHaveBeenCalledWith(mockSetState);
    expect(wrapper.find(MockComponent)).toHaveProp('data', contentData);
  });

  it('renders using the bannerData prop for getting data', () => {
    const { wrapper } = setup({
      staticBannerData: bannerData,
      updateBannerData: undefined,
    });

    expect(wrapper.find(BannerBackground)).toHaveProp({
      backgroundImageSrc: bannerData.backgroundImageSrc,
    });
    expect(wrapper.find(BannerCategoryImage)).toHaveProp({
      boxArtURL: bannerData.categoryImageSrc,
    });
    expect(wrapper.find('Layout[padding=3]')).toExist();

    expect(wrapper.find(MockComponent)).toHaveProp('data', contentData);
  });

  it('renders without category image and without contentRenderer', () => {
    const { wrapper } = setup({
      BannerContent: undefined,
      showCategoryImage: false,
    });

    expect(wrapper.find(BannerBackground)).toExist();
    expect(wrapper.find(BannerCategoryImage)).not.toExist();
    const bannerContentContainer = wrapper.find('Layout[padding=0]');

    expect(bannerContentContainer).toExist();
    expect(bannerContentContainer.children()).not.toExist();
  });
});
