import type { FC } from 'react';
import styled from 'styled-components';
import { getSrc } from 'tachyon-more-ui';
import { Layout } from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import { OVERSCAN_PADDING_REM, TOP_NAV_HEIGHT_REM } from '../../../../config';
import type { FadeInProps } from '../../styleMixins';
import { fadeIn } from '../../styleMixins';

export type BannerBackgroundProps = FadeInProps & {
  backgroundImageSrc: Parameters<typeof getSrc>[0];
};

// intentionally request a screen-filling image which we then align to the top
// of the banner element and clip off the bottom
const BACKGROUND_IMAGE_ASPECT_RATIO = 9 / 16;
// request a half-resolution (@1080p) image for the banner to save on size with
// minimal visual impact since we put half-transparent black overlay on top of it
const BACKGROUND_IMAGE_WIDTH = 540;

// istanbul ignore next: trivial
/**
 * Adds a CSS background image from a url() with a dark overlay on top of it
 */
function getBackgroundImage({
  backgroundImageSrc,
}: BannerBackgroundProps): string {
  if (!backgroundImageSrc) {
    return '';
  }

  return `
    background-image: linear-gradient(to bottom, ${
      DarkThemeMap['color-background-overlay']
    } 50%, ${DarkThemeMap['color-background-body']}), url('${getSrc(
    backgroundImageSrc,
    BACKGROUND_IMAGE_WIDTH,
    BACKGROUND_IMAGE_ASPECT_RATIO,
  )}');
  `;
}

// istanbul ignore next: trivial
// Height is set based on the total height from the top of the page to the
// bottom of the banner
// background-color is always set in order to prevent bleed-through while the
// image is loading.
const ScBannerBackground = styled.div<BannerBackgroundProps>`
  background-color: ${DarkThemeMap['color-background-body']};
  background-size: 100vw;
  box-sizing: content-box;
  height: 20rem;
  ${fadeIn}
  ${getBackgroundImage}
  padding-left: ${OVERSCAN_PADDING_REM}rem;
  padding-right: ${OVERSCAN_PADDING_REM}rem;
  padding-top: ${TOP_NAV_HEIGHT_REM}rem;
`;

// Necessary to protect against background bleed while the banner's image/text
// are fading in
const ScBackdrop = styled.div`
  background-color: ${DarkThemeMap['color-background-body']};
`;

// istanbul ignore next: trivial
export const BannerBackground: FC<BannerBackgroundProps> = ({
  backgroundImageSrc,
  children,
}) => (
  <ScBackdrop>
    <ScBannerBackground backgroundImageSrc={backgroundImageSrc}>
      <Layout fullHeight padding={{ x: 1, y: 2 }}>
        {children}
      </Layout>
    </ScBannerBackground>
  </ScBackdrop>
);

BannerBackground.displayName = 'BannerBackground';
