import { useRef } from 'react';
import type {
  BannerData,
  BannerDataUpdateFn,
  UpdateBannerData,
} from '../Banner';

// This is basically a tiny purpose-built `useObservable`, with the initial data
// and normal function invocation creating a clamped event stream and `__setRef`
// equivalent to a stream subscription. This could be replaced with recoil or
// rxjs or any other `useObservable`-esque implementations we adopt in the
// future. Currently, the `__getInitialData` and `__setRef` functions are
// intended as "private" functions only to be used by the `Banner` component.

/**
 * This is a function that enables communication from a component in a different
 * subtree to the Banner component without having to engage any parent
 * components or contexts, thus avoiding unnecessary rerenders. The invocation
 * loads the initial data, and it returns an update function that can be called
 * to transmit new data to the Banner component. It must be passed to the
 * `BannerPage` component (under the updateBannerData prop) in order to wire up
 * proper communication.
 */
export function useBannerData<T>(
  initialData: BannerData<T>,
): UpdateBannerData<T> {
  const ref = useRef<BannerDataUpdateFn<T>>(() => undefined);

  function updateBannerData(data: BannerData<T>): void {
    ref.current(data);
  }

  updateBannerData.__setRef = (updater: BannerDataUpdateFn<T>) => {
    ref.current = updater;
  };

  updateBannerData.__getInitialData = () => initialData;

  return updateBannerData;
}
