import type { FC } from 'react';
import {
  createContext,
  useContext,
  useEffect,
  useMemo,
  useRef,
  useState,
} from 'react';
import { NodeNav, VerticalNav } from 'tachyon-tv-nav';
import {
  Background,
  BorderRadius,
  Display,
  FlexDirection,
  Layout,
  Position,
} from 'twitch-core-ui';
import type { FocusableIconButtonProps } from '../FocusableIconButton';
import { FocusableIconButton } from '../FocusableIconButton';

type FlyOutMenuProps = Omit<FocusableIconButtonProps, 'focusIndex'> & {
  elementCount: number;
  focusIndex: number;
  initialChildFocusIndex?: number;
};

type FlyOutControls = {
  setShowFlyout: (shouldShowFlyout: boolean) => void;
};

const flyOutContext = createContext<FlyOutControls>({
  setShowFlyout: () => undefined,
});

export const useFlyOutControls = (): FlyOutControls => {
  return useContext(flyOutContext);
};

/**
 * Component used to show a flyout menu when the button is clicked and hide
 * it when the user navigates away from the menu.
 *
 * @example
 * <FlyOutMenu { ...flyOutMenuProps }>
 *   <ChildComponent />
 * </FlyOutMenu>
 *
 * const ChildComponent = () => {
 *   const { setShowFlyout } = useFlyOutControls();
 *   return (
 *    <FocusableTextButton
 *      focusIndex={0}
 *      onClick={() => {
 *       setShowFlyout(false);
 *      }}
 *    >
 *      Click Here!
 *    </FocusableTextButton>
 *   );
 * }
 */
export const FlyOutMenu: FC<FlyOutMenuProps> = ({
  children,
  elementCount,
  focusIndex,
  initialChildFocusIndex,
  ...baseButtonProps
}) => {
  const ref = useRef<HTMLDivElement>(null);
  const [showFlyout, setShowFlyout] = useState(false);
  const ctx = useMemo(() => ({ setShowFlyout }), []);

  useEffect(() => {
    if (!showFlyout) {
      return;
    }

    // dismiss the flyout if the user clicks/wands elsewhere on the screen
    const clickHandler = (e: MouseEvent) => {
      if (!ref.current?.contains(e.target as Node)) {
        setShowFlyout(false);
      }
    };

    document.addEventListener('click', clickHandler);

    return () => {
      document.removeEventListener('click', clickHandler);
    };
  }, [showFlyout]);

  const dismissOnNavOut = () => {
    setShowFlyout(false);
  };

  return (
    <flyOutContext.Provider value={ctx}>
      <Layout position={Position.Relative}>
        <VerticalNav elementCount={showFlyout ? 2 : 1} focusIndex={focusIndex}>
          {showFlyout && (
            // this NodeNav prevents the user from naving out of the flyout vertically
            <NodeNav
              focusIndex={0}
              onDown={dismissOnNavOut}
              onUp={() => {
                // Disable normal nav behavior by returning true
                return true;
              }}
            >
              {/* onLeft and onRight cause the dialog to dismiss when the user navs away left/right */}
              <VerticalNav
                elementCount={elementCount}
                focusIndex={0}
                initialChildFocusIndex={initialChildFocusIndex}
                onLeft={dismissOnNavOut}
                onRight={dismissOnNavOut}
                takeFocusOnFirstRender
              >
                <Layout
                  attachBottom
                  background={Background.Base}
                  border
                  borderRadius={BorderRadius.Large}
                  display={Display.Flex}
                  flexDirection={FlexDirection.Column}
                  margin={{ bottom: 4 }}
                  position={Position.Absolute}
                  ref={ref}
                >
                  {children}
                </Layout>
              </VerticalNav>
            </NodeNav>
          )}
          <FocusableIconButton
            focusIndex={showFlyout ? 1 : 0}
            onClick={() => {
              setShowFlyout(true);
            }}
            {...baseButtonProps}
          />
        </VerticalNav>
      </Layout>
    </flyOutContext.Provider>
  );
};

FlyOutMenu.displayName = 'FlyOutMenu';
