import type { FC } from 'react';
import styled from 'styled-components';
import { useFocus } from 'tachyon-tv-nav';
import type { CardConfig } from '../../../config';
import { CARD_ANIMATION } from '../../../config';

type ScFocusableCardImageContainerProps = {
  /**
   * Use to round the box shadow effect for rounded card images such as Avatars.
   * Value should match the image width in REM.
   */
  borderRadiusRem?: number;
  config: CardConfig;
  focused: boolean;
};

function animate({
  borderRadiusRem,
  config,
  focused,
}: ScFocusableCardImageContainerProps): string {
  if (!focused) {
    return '';
  }

  // "horizontalPaddingRem" and the vertical translation value must
  // remain 1:1 to maintain the symmetrical corner effect
  return `
    border-radius: ${borderRadiusRem ?? 0}rem;
    box-shadow: 0px 12px 32px rgba(0, 0, 0, 0.35), 0px 0px 8px rgba(0, 0, 0, 0.35);
    transform: translate3d(
      0,
      -${config.horizontalPaddingRem}rem,
      0
    );
  `;
}

export const ScFocusableCardImageContainer = styled.div<ScFocusableCardImageContainerProps>`
  ${animate}
  transition: transform ${CARD_ANIMATION};
`;

type FocusableCardImageContainerProps = Omit<
  ScFocusableCardImageContainerProps,
  'focused'
> & {
  focusIndex: number;
};

// istanbul ignore next: trivial
export const FocusableCardImageContainer: FC<FocusableCardImageContainerProps> =
  ({ focusIndex, ...baseProps }) => {
    const { focused } = useFocus(focusIndex);
    return <ScFocusableCardImageContainer focused={focused} {...baseProps} />;
  };

FocusableCardImageContainer.displayName = 'FocusableCardImageContainer';
