import type { FC } from 'react';
import { useEffect } from 'react';
import styled from 'styled-components';
import { refFocusHandler, useFocus } from 'tachyon-tv-nav';
import { usePrevious } from 'tachyon-utils';
import type { CoreLinkProps, MarginValue } from 'twitch-core-ui';
import { Background, CoreLink, ThemeRegion } from 'twitch-core-ui';
import type { CardConfig } from '../../../config';
import {
  CARD_ANIMATION,
  CARD_HEIGHT_REM,
  STREAM_CARD_AVATAR_SIZE_REM,
} from '../../../config';
import type { FocusableStyleProps } from '../styleMixins';
import { focusableBorderStyle } from '../styleMixins';

type ScCardLinkContainerProps = FocusableStyleProps & {
  $config: CardConfig;
};

// We override MediaCard image flex-basis for Chrome 38 compatibility. We do it
// here for perf reasons so as to avoid having to make an s-c div solely for
// that purpose closer to the composed MediaCard component.
export const ScCardLinkContainer = styled.div<ScCardLinkContainerProps>`
  height: ${CARD_HEIGHT_REM}rem;
  ${focusableBorderStyle}
  padding: 0 ${({ $config }) => $config.horizontalPaddingRem}rem;
  transition: background-color ${CARD_ANIMATION};
  width: ${({ $config }) => $config.widthRem}rem;

  & .tw-media-card-meta__image {
    flex-basis: ${STREAM_CARD_AVATAR_SIZE_REM}rem;
  }
`;

export type FocusableCardLinkProps = {
  config: CardConfig;
  focusIndex: number;
  marginBottom?: MarginValue;
  onClick?: () => void;
  onFocus?: (focusIndex: number) => void;
  renderLink: CoreLinkProps['renderLink'];
};

// istanbul ignore next: trivial
export const FocusableCardLink: FC<FocusableCardLinkProps> = ({
  children,
  config,
  focusIndex,
  marginBottom,
  onClick,
  onFocus,
  renderLink,
}) => {
  const { focused, takeFocus } = useFocus(focusIndex);
  const prevFocused = usePrevious(focused);

  useEffect(() => {
    if (onFocus && focused && !prevFocused) {
      onFocus(focusIndex);
    }
  }, [focusIndex, focused, onFocus, prevFocused]);

  return (
    <ThemeRegion
      background={focused ? Background.Alt : Background.Inherit}
      margin={{ bottom: marginBottom }}
      theme={focused ? 'light' : 'dark'}
    >
      <CoreLink
        linkTo="/deferToRenderLink"
        onClick={onClick}
        onMouseEnter={takeFocus}
        refHandler={refFocusHandler(focused)}
        renderLink={renderLink}
      >
        <ScCardLinkContainer $config={config} $focused={focused}>
          {children}
        </ScCardLinkContainer>
      </CoreLink>
    </ThemeRegion>
  );
};

FocusableCardLink.displayName = 'FocusableCardLink';
