import type { FC } from 'react';
import { memo } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { formatLiveViewersCount, getBoxArtSrcAndSrcSet } from 'tachyon-more-ui';
import { getGameDisplayName } from 'tachyon-utils';
import {
  AspectRatio,
  CardImage,
  Color,
  CoreText,
  MediaCard,
  MediaCardImage,
  MediaCardMeta,
} from 'twitch-core-ui';
import { CATEGORY_CARD_CONFIG } from '../../../config';
import { RouteName, renderTachyonLink } from '../../../routing';
import { FocusableCardImageContainer } from '../FocusableCardImageContainer';
import type { FocusableCardLinkProps } from '../FocusableCardLink';
import { FocusableCardLink } from '../FocusableCardLink';
import type { FocusableCategoryCard_category } from './__generated__/FocusableCategoryCard_category.graphql';

type CategoryCardBaseProps = {
  category: FocusableCategoryCard_category;
  focusIndex: number;
};

// Ensure that props provided to this component remain stable to avoid expensive
// re-renders. Also avoid using context-hooks directly in here for the same
// reason; instead push them into child components like FocusableCardImageContainer.
// istanbul ignore next: trivial
const CategoryCardBase: FC<CategoryCardBaseProps> = ({
  category,
  focusIndex,
}) => {
  const { formatMessage, formatNumberShort } = useIntl();
  const name = getGameDisplayName(category);

  return (
    <MediaCard
      image={
        <FocusableCardImageContainer
          config={CATEGORY_CARD_CONFIG}
          focusIndex={focusIndex}
        >
          <MediaCardImage
            image={
              <CardImage
                alt=""
                aspect={AspectRatio.BoxArt}
                sizes={[{ size: `${CATEGORY_CARD_CONFIG.widthRem}rem` }]}
                {...getBoxArtSrcAndSrcSet(category.boxArtURL)}
              />
            }
            ratio={AspectRatio.BoxArt}
          />
        </FocusableCardImageContainer>
      }
      meta={
        <MediaCardMeta
          links={
            category.viewersCount !== null && (
              <CoreText
                children={formatLiveViewersCount(
                  formatMessage,
                  category.viewersCount,
                  formatNumberShort(category.viewersCount),
                )}
                color={Color.Base}
                ellipsis
              />
            )
          }
          title={name}
        />
      }
    />
  );
};

CategoryCardBase.displayName = 'CategoryCardBase';

const CategoryCard = memo(CategoryCardBase);

export type FocusableCategoryCardProps = Pick<
  FocusableCardLinkProps,
  'focusIndex' | 'marginBottom' | 'onClick' | 'onFocus'
> & {
  category: FocusableCategoryCard_category;
};

// istanbul ignore next: trivial
export const FocusableCategoryCardBase: FC<FocusableCategoryCardProps> = ({
  category,
  focusIndex,
  marginBottom,
  onClick,
  onFocus,
}) => {
  return (
    <FocusableCardLink
      config={CATEGORY_CARD_CONFIG}
      focusIndex={focusIndex}
      marginBottom={marginBottom}
      onClick={onClick}
      onFocus={onFocus}
      renderLink={renderTachyonLink({
        route: RouteName.GameDirectory,
        routeParams: { gameAlias: category.name },
      })}
    >
      <CategoryCard category={category} focusIndex={focusIndex} />
    </FocusableCardLink>
  );
};

FocusableCategoryCardBase.displayName = 'FocusableCategoryCardBase';

export const FocusableCategoryCard = createFragmentContainer(
  FocusableCategoryCardBase,
  {
    category: graphql`
      fragment FocusableCategoryCard_category on Game {
        name
        id
        name
        displayName
        viewersCount
        boxArtURL
      }
    `,
  },
);
