import type { FC } from 'react';
import type { CSSObject } from 'styled-components';
import styled from 'styled-components';
import type { InteractionDataPayload } from 'tachyon-event-tracker';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { refFocusHandler, useFocus } from 'tachyon-tv-nav';
import type { CoreButtonProps } from 'twitch-core-ui';
import { CoreButton, CoreButtonType } from 'twitch-core-ui';
import { DarkThemeMap, LightThemeMap } from 'twitch-core-ui-tokens';
import type { FocusableStyleProps } from '../styleMixins';
import { focusableBorderStyle } from '../styleMixins';

export const DEFAULT_FOCUSED_TEXT_COLOR = LightThemeMap['color-text-alt'];
export const DEFAULT_UNFOCUSED_TEXT_COLOR = DarkThemeMap['color-text-alt'];
export const DEFAULT_DESTRUCTIVE_TEXT_COLOR =
  LightThemeMap['color-text-button-destructive'];

type FocusedBackgroundColorProps = {
  $focused: boolean;
  $focusedType: CoreButtonType;
  $withUnfocusBackground?: boolean;
};

function focusedBackgroundColor({
  $focused,
  $focusedType,
  $withUnfocusBackground,
}: FocusedBackgroundColorProps): CSSObject {
  if ($focused) {
    switch ($focusedType) {
      case CoreButtonType.Primary:
      case CoreButtonType.Secondary:
      case CoreButtonType.Text:
        return {
          'background-color':
            DarkThemeMap['color-background-interactable-overlay-selected'],
          color: DEFAULT_FOCUSED_TEXT_COLOR,
        };
      case CoreButtonType.Destructive:
        return {
          'background-color':
            LightThemeMap['color-background-button-destructive-default'],
          color: DEFAULT_DESTRUCTIVE_TEXT_COLOR,
        };
      default:
        return CoreButtonType as never;
    }
  }

  if ($withUnfocusBackground) {
    return {
      'background-color':
        DarkThemeMap['color-background-interactable-overlay-active'],
    };
  }

  return {
    'background-color':
      DarkThemeMap['color-background-button-overlay-text-default'],
  };
}

type ScCoreButtonProps = FocusableStyleProps &
  FocusedBackgroundColorProps &
  Omit<CoreButtonProps, 'type'> & {
    $contentLeft?: boolean;
    $withUnfocusBackground?: boolean;
  };

/**
 * The underlying CoreButton is set to type "text" for its transparent background.
 *
 * The cursor enabled / not hovered rule is used to prevent the non-transparent
 * focus style from applying which can result in two focused components on the page.
 * In the future, we might always adjust focus on hover so we could get rid of this
 * weird edge case.
 */
export const ScCoreButton = styled(CoreButton)<ScCoreButtonProps>`
  color: ${DEFAULT_UNFOCUSED_TEXT_COLOR};
  ${focusableBorderStyle}
  ${focusedBackgroundColor}
  ${({ $contentLeft }) =>
    $contentLeft && 'justify-content: flex-start !important;'}

  :hover {
    color: ${({ $focused, $focusedType }) => {
      if ($focusedType === CoreButtonType.Destructive) {
        return DEFAULT_DESTRUCTIVE_TEXT_COLOR;
      }
      // If the user mixes inputs, we can get into a state where the hover state
      // is active but the button does not have focus. In these instances we
      // want to maintain the light colored text for readability.
      return $focused
        ? DEFAULT_FOCUSED_TEXT_COLOR
        : DEFAULT_UNFOCUSED_TEXT_COLOR;
    }};
  }
`;

export type FocusableCoreButtonProps = Omit<CoreButtonProps, 'type'> &
  Partial<Omit<InteractionDataPayload, 'interaction'>> & {
    contentLeft?: boolean;
    focusIndex: number;
    focusedType?: CoreButtonType;
    withUnfocusBackground?: boolean;
  };

export const FocusableCoreButton: FC<FocusableCoreButtonProps> = ({
  children,
  contentLeft,
  focusIndex,
  focusedType = CoreButtonType.Primary,
  interactionContent,
  interactionTargetPath,
  size,
  withUnfocusBackground,
  ...buttonProps
}) => {
  const { focused, takeFocus } = useFocus(focusIndex);

  const trackInteraction = useInteractionTracking();

  function onClick(): void {
    // Only report event if this is a button and not rendering a link. Otherwise we will cause double
    // reporting on buttons that are links, due to this:
    // https://git.xarth.tv/emerging-platforms/tachyon/blob/main/apps/starshot/src/routing/components/TachyonLink/index.tsx#L57-L63
    if (interactionContent && !buttonProps.renderLink) {
      trackInteraction({
        interaction: InteractionType.Click,
        interactionContent,
        interactionTargetPath,
      });
    }
  }

  return (
    <ScCoreButton
      $contentLeft={contentLeft}
      $focused={focused}
      $focusedType={focusedType}
      $withUnfocusBackground={withUnfocusBackground}
      children={children}
      onClick={onClick}
      onMouseEnter={takeFocus}
      refHandler={refFocusHandler(focused)}
      {...buttonProps}
    />
  );
};
FocusableCoreButton.displayName = 'FocusableCoreButton';
