import { act } from '@testing-library/react-hooks';
import { useInteractionTracking } from 'tachyon-event-tracker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { useFocus } from 'tachyon-tv-nav';
import { CoreButtonType } from 'twitch-core-ui';
import {
  DEFAULT_DESTRUCTIVE_TEXT_COLOR,
  DEFAULT_FOCUSED_TEXT_COLOR,
  DEFAULT_UNFOCUSED_TEXT_COLOR,
  FocusableCoreButton,
  ScCoreButton,
} from '.';

jest.mock('tachyon-event-tracker', () => ({
  ...jest.requireActual('tachyon-event-tracker'),
  useInteractionTracking: jest.fn(),
}));

jest.mock('tachyon-tv-nav', () => ({
  ...jest.requireActual('tachyon-tv-nav'),
  useFocus: jest.fn(() => ({ focused: true })),
}));

const mockUseInteractionTracking = useInteractionTracking as jest.Mock;
const mockUseFocus = useFocus as jest.Mock;
const MockChildren = () => <div />;

describe(FocusableCoreButton, () => {
  const setup = createMountWrapperFactory(FocusableCoreButton, () => ({
    children: <MockChildren />,
    focusIndex: 0,
  }));

  it('tracks interactions when interactionContent is provided', () => {
    const mockTrackInteraction = jest.fn();
    mockUseInteractionTracking.mockImplementationOnce(
      () => mockTrackInteraction,
    );
    const { wrapper } = setup({ interactionContent: 'foo' });

    act(() => {
      wrapper.find(ScCoreButton).prop('onClick')!();
    });

    expect(mockTrackInteraction).toHaveBeenCalledTimes(1);
  });

  it('displays correctly when hovered and focused', () => {
    mockUseFocus.mockReturnValue({ focused: true });
    const { wrapper } = setup();

    const button = wrapper.find(ScCoreButton);
    expect(button).toHaveProp('$focused', true);
    expect(button).toHaveStyleRule('color', DEFAULT_FOCUSED_TEXT_COLOR, {
      modifier: ':hover',
    });
  });

  it('displays correctly when hovered but not focused', () => {
    mockUseFocus.mockReturnValue({ focused: false });
    const { wrapper } = setup();

    const button = wrapper.find(ScCoreButton);
    expect(button).toHaveProp('$focused', false);
    expect(button).toHaveStyleRule('color', DEFAULT_UNFOCUSED_TEXT_COLOR, {
      modifier: ':hover',
    });
  });

  it('displays correctly when unfocused and button type is destructive', () => {
    mockUseFocus.mockReturnValue({
      focused: false,
    });
    const { wrapper } = setup({ focusedType: CoreButtonType.Destructive });

    const button = wrapper.find(ScCoreButton);
    expect(button).toHaveProp('$focused', false);
    expect(button).toHaveStyleRule('color', DEFAULT_UNFOCUSED_TEXT_COLOR);
    expect(button).toHaveStyleRule('color', DEFAULT_DESTRUCTIVE_TEXT_COLOR, {
      modifier: ':hover',
    });
  });

  it('displays correctly when focused and button type is destructive', () => {
    mockUseFocus.mockReturnValue({
      focused: true,
    });
    const { wrapper } = setup({ focusedType: CoreButtonType.Destructive });

    const button = wrapper.find(ScCoreButton);
    expect(button).toHaveProp('$focused', true);
    expect(button).toHaveStyleRule('color', DEFAULT_DESTRUCTIVE_TEXT_COLOR);
    expect(button).toHaveStyleRule('color', DEFAULT_DESTRUCTIVE_TEXT_COLOR, {
      modifier: ':hover',
    });
  });
});
