import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useCurrentUser } from 'tachyon-auth';
import { useIntl } from 'tachyon-intl';
import { SVGAsset } from 'twitch-core-ui';
import { useFollow } from '../../../hooks';
import {
  RouteName,
  createRedirectQuery,
  renderTachyonLink,
} from '../../../routing';
import type { FocusableCoreButtonProps } from '../FocusableCoreButton';
import { FocusableIconButton } from '../FocusableIconButton';
import { FocusableTextButton } from '../FocusableTextButton';
import type { FocusableFollowButton_channel } from './__generated__/FocusableFollowButton_channel.graphql';

export type FocusableFollowButtonProps = FocusableCoreButtonProps & {
  channel: FocusableFollowButton_channel;
  iconOnly?: true;
  route: RouteName;
};

export const FocusableFollowButtonBase: FC<FocusableFollowButtonProps> = ({
  channel,
  focusIndex,
  iconOnly,
  route,
  withUnfocusBackground,
}) => {
  const { formatMessage } = useIntl();
  const { loggedIn } = useCurrentUser();
  const { follow, following, unfollow } = useFollow(channel);

  const actionText = following
    ? formatMessage('Unfollow', 'ChannelProfile')
    : formatMessage('Follow', 'ChannelProfile');

  const icon = following ? SVGAsset.Heart : SVGAsset.FollowHollow;

  const interactionButtonProps: Partial<FocusableCoreButtonProps> = loggedIn
    ? {
        onClick: following ? unfollow : follow,
      }
    : {
        linkTo: '/deferToRenderLink',
        renderLink: renderTachyonLink({
          query: createRedirectQuery({
            route,
            routeParams: { login: channel.login },
          }),
          route: RouteName.Login,
        }),
      };

  if (iconOnly) {
    return (
      <FocusableIconButton
        aria-label={actionText}
        focusIndex={focusIndex}
        icon={icon}
        withUnfocusBackground={withUnfocusBackground}
        {...interactionButtonProps}
      />
    );
  }

  return (
    <FocusableTextButton
      focusIndex={focusIndex}
      icon={icon}
      withUnfocusBackground={withUnfocusBackground}
      {...interactionButtonProps}
    >
      {actionText}
    </FocusableTextButton>
  );
};

FocusableFollowButtonBase.displayName = 'FocusableFollowButton';

export const FocusableFollowButton = createFragmentContainer(
  FocusableFollowButtonBase,
  {
    channel: graphql`
      fragment FocusableFollowButton_channel on User {
        login
        ...useFollow_targetChannel @relay(mask: false)
      }
    `,
  },
);
