import Faker from 'faker';
import { act } from 'react-dom/test-utils';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { SVGAsset } from 'twitch-core-ui';
import { RouteName } from '../../../routing';
import { FocusableIconButton } from '../FocusableIconButton';
import { FocusableTextButton } from '../FocusableTextButton';
import { FocusableFollowButtonBase } from '.';

const mockUseCurrentUser = { loggedIn: true };
jest.mock('tachyon-auth', () => ({
  ...jest.requireActual('tachyon-auth'),
  useCurrentUser: () => mockUseCurrentUser,
}));

let mockFollowing = true;
const mockFollow = jest.fn();
const mockUnfollow = jest.fn();

jest.mock('../../../hooks', () => ({
  useFollow: () => ({
    follow: mockFollow,
    following: mockFollowing,
    unfollow: mockUnfollow,
  }),
}));

describe('FollowButton', () => {
  beforeEach(() => {
    jest.resetAllMocks();
  });

  const setup = createShallowWrapperFactory(FocusableFollowButtonBase, () => ({
    channel: {
      ' $refType': 'FocusableFollowButton_channel',
      id: Faker.datatype.uuid(),
      login: 'therealderekt',
      self: {
        follower: {
          followedAt: 'Noon',
        },
      },
    },
    focusIndex: 0,
    route: RouteName.Channel,
  }));

  it('shows the empty heart when not followed', () => {
    mockFollowing = false;
    const { wrapper } = setup();
    expect(wrapper.find(FocusableTextButton)).toHaveProp({
      icon: SVGAsset.FollowHollow,
    });
  });

  it('shows the filled in heart when following', () => {
    mockFollowing = true;
    const { wrapper } = setup();
    expect(wrapper.find(FocusableTextButton)).toHaveProp({
      icon: SVGAsset.Heart,
    });
  });

  it('renders as FocusableTextButton by default', () => {
    const { wrapper } = setup();
    expect(wrapper.debug()).toContain('Unfollow');
    expect(wrapper.find(FocusableTextButton)).toHaveProp({
      icon: SVGAsset.Heart,
    });
  });

  it('renders as a FocusableIconButton in icon mode', () => {
    const { wrapper } = setup({ iconOnly: true });
    expect(wrapper.debug()).toContain('Unfollow');
    expect(wrapper.find(FocusableIconButton)).toHaveProp({
      'aria-label': 'Unfollow',
      icon: SVGAsset.Heart,
    });
  });

  describe('when user is logged in', () => {
    beforeEach(() => {
      mockUseCurrentUser.loggedIn = true;
    });

    it('does not link to the login page as a text button', () => {
      const { wrapper } = setup();
      expect(wrapper.find(FocusableTextButton)).not.toHaveProp('renderLink');
    });

    it('does not link to the login page as an icon button', () => {
      const { wrapper } = setup({ iconOnly: true });
      expect(wrapper.find(FocusableTextButton)).not.toHaveProp('renderLink');
    });

    it('executes unfollow mutation when unfollow button is clicked', () => {
      mockFollowing = true;
      const { wrapper } = setup();
      expect(wrapper.debug()).toContain('Unfollow');

      expect(mockUnfollow).not.toHaveBeenCalled();
      act(() => {
        wrapper.find(FocusableTextButton).simulate('click');
      });

      expect(mockUnfollow).toHaveBeenCalledTimes(1);
    });

    it('executes follow mutation when follow button is clicked', () => {
      mockFollowing = false;
      const { wrapper } = setup();
      expect(wrapper.debug()).toContain('Follow');

      expect(mockFollow).not.toHaveBeenCalled();
      act(() => {
        wrapper.find(FocusableTextButton).simulate('click');
      });

      expect(mockFollow).toHaveBeenCalledTimes(1);
    });
  });

  describe('when user is logged out', () => {
    beforeEach(() => {
      mockUseCurrentUser.loggedIn = false;
    });

    it('renders as a text button with a link to the login page', () => {
      const { wrapper } = setup();
      expect(wrapper.find(FocusableTextButton)).toHaveProp('renderLink');
      expect(wrapper.find(FocusableTextButton)).not.toHaveProp('onClick');
    });

    it('renders as an icon button with a link to the login page', () => {
      const { wrapper } = setup({ iconOnly: true });
      expect(wrapper.find(FocusableIconButton)).toHaveProp('renderLink');
      expect(wrapper.find(FocusableIconButton)).not.toHaveProp('onClick');
    });
  });
});
