import type { FC } from 'react';
import { memo } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { Avatar, formatFollowers, formatLastLive } from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  Color,
  CoreText,
  Display,
  FontSize,
  JustifyContent,
  Layout,
  MediaCard,
  MediaCardMeta,
  TextAlign,
  TextType,
} from 'twitch-core-ui';
import {
  OFFLINE_CHANNEL_AVATAR_SIZE_REM,
  OFFLINE_CHANNEL_CARD_CONFIG,
} from '../../../config';
import { RouteName, renderTachyonLink } from '../../../routing';
import { FocusableCardImageContainer } from '../FocusableCardImageContainer';
import type { FocusableCardLinkProps } from '../FocusableCardLink';
import { FocusableCardLink } from '../FocusableCardLink';
import type { FocusableOfflineChannelCard_channel } from './__generated__/FocusableOfflineChannelCard_channel.graphql';

type OfflineChannelCardBaseProps = {
  channel: FocusableOfflineChannelCard_channel;
  focusIndex: number;
};

// Ensure that props provided to this component remain stable to avoid expensive
// re-renders. Also avoid using context-hooks directly in here for the same
// reason; instead push them into child components like FocusableCardImageContainer.
const OfflineChannelCardBase: FC<OfflineChannelCardBaseProps> = ({
  channel,
  focusIndex,
}) => {
  const intl = useIntl();
  const name = getFormattedUserDisplayName(channel);

  return (
    <MediaCard
      image={
        <Layout display={Display.Flex} justifyContent={JustifyContent.Center}>
          <FocusableCardImageContainer
            borderRadiusRem={OFFLINE_CHANNEL_AVATAR_SIZE_REM}
            config={OFFLINE_CHANNEL_CARD_CONFIG}
            focusIndex={focusIndex}
          >
            <Avatar
              alt=""
              sizeRem={OFFLINE_CHANNEL_AVATAR_SIZE_REM}
              src={channel.profileImageURL}
              userLogin={channel.login}
            />
          </FocusableCardImageContainer>
        </Layout>
      }
      meta={
        <MediaCardMeta
          links={
            <Layout textAlign={TextAlign.Center}>
              {channel.lastBroadcast?.startedAt && (
                <CoreText color={Color.Base} ellipsis>
                  {formatLastLive(intl, channel.lastBroadcast.startedAt)}
                </CoreText>
              )}
              {channel.followers?.totalCount && (
                <CoreText color={Color.Base} ellipsis>
                  {formatFollowers(intl, channel.followers.totalCount)}
                </CoreText>
              )}
            </Layout>
          }
          title={
            <Layout textAlign={TextAlign.Center}>
              <CoreText
                color={Color.Base}
                ellipsis
                fontSize={FontSize.Size5}
                type={TextType.H3}
              >
                {name}
              </CoreText>
            </Layout>
          }
        />
      }
    />
  );
};

OfflineChannelCardBase.displayName = 'OfflineChannelCardBase';

const OfflineChannelCard = memo(OfflineChannelCardBase);

type FocusableOfflineChannelCardProps = Pick<
  FocusableCardLinkProps,
  'focusIndex' | 'onClick' | 'onFocus'
> & {
  channel: FocusableOfflineChannelCard_channel;
};

// istanbul ignore next: trivial
export const FocusableOfflineChannelCardBase: FC<FocusableOfflineChannelCardProps> =
  ({ channel, focusIndex, onClick, onFocus }) => {
    const renderLink = channel.login
      ? renderTachyonLink({
          route: RouteName.Channel,
          routeParams: { login: channel.login },
        })
      : renderTachyonLink({
          route: RouteName.NotFound,
        });

    return (
      <FocusableCardLink
        config={OFFLINE_CHANNEL_CARD_CONFIG}
        focusIndex={focusIndex}
        onClick={onClick}
        onFocus={onFocus}
        renderLink={renderLink}
      >
        <OfflineChannelCard channel={channel} focusIndex={focusIndex} />
      </FocusableCardLink>
    );
  };

FocusableOfflineChannelCardBase.displayName = 'FocusableOfflineChannelCardBase';

export const FocusableOfflineChannelCard = createFragmentContainer(
  FocusableOfflineChannelCardBase,
  {
    channel: graphql`
      fragment FocusableOfflineChannelCard_channel on User {
        displayName
        followers {
          totalCount
        }
        lastBroadcast {
          startedAt
        }
        login
        profileImageURL(width: 300)
      }
    `,
  },
);
