import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useTrackedListItem } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import {
  Avatar,
  StreamTypeIndicator,
  formatLiveViewersCount,
  getChannelThumbnailSrcAndSrcSet,
  getStreamType,
} from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AspectRatio,
  CardImage,
  Color,
  CoreText,
  MediaCard,
  MediaCardImage,
  MediaCardMeta,
  MediaCardStat,
} from 'twitch-core-ui';
import {
  STREAM_CARD_AVATAR_SIZE_REM,
  WATCHABLE_CARD_CONFIG,
} from '../../../config';
import { RouteName, renderTachyonLink } from '../../../routing';
import { FocusableCardImageContainer } from '../FocusableCardImageContainer';
import type { FocusableCardLinkProps } from '../FocusableCardLink';
import { FocusableCardLink } from '../FocusableCardLink';
import type { FocusableStreamCard_stream } from './__generated__/FocusableStreamCard_stream.graphql';

type StreamCardBaseProps = {
  focusIndex: number;
  stream: FocusableStreamCard_stream;
};

// istanbul ignore next: trivial
// Ensure that props provided to this component remain stable to avoid expensive
// re-renders. Also avoid using context-hooks directly in here for the same
// reason; instead push them into child components like FocusableCardImageContainer.
const StreamCardBase: FC<StreamCardBaseProps> = ({ focusIndex, stream }) => {
  const { formatMessage, formatNumberShort } = useIntl();
  const broadcaster = stream.broadcaster;
  const streamType = getStreamType(stream);

  return (
    <MediaCard
      image={
        <FocusableCardImageContainer
          config={WATCHABLE_CARD_CONFIG}
          focusIndex={focusIndex}
        >
          <MediaCardImage
            bottomLeft={
              stream.viewersCount !== null && (
                <MediaCardStat>
                  {formatLiveViewersCount(
                    formatMessage,
                    stream.viewersCount,
                    formatNumberShort(stream.viewersCount, {
                      maximumSignificantDigits: 2,
                    }),
                  )}
                </MediaCardStat>
              )
            }
            image={
              <CardImage
                alt=""
                aspect={AspectRatio.Aspect16x9}
                sizes={[{ size: `${WATCHABLE_CARD_CONFIG.widthRem}rem` }]}
                {...getChannelThumbnailSrcAndSrcSet(stream.previewImageURL)}
              />
            }
            ratio={AspectRatio.Aspect16x9}
            topLeft={
              <StreamTypeIndicator
                isOverlay
                isPulsing={false}
                streamType={streamType}
              />
            }
          />
        </FocusableCardImageContainer>
      }
      meta={
        <MediaCardMeta
          image={
            <Avatar
              alt=""
              sizeRem={
                STREAM_CARD_AVATAR_SIZE_REM /* Note that this size also relies on a MediaCard style override in FocusableCardLink */
              }
              src={broadcaster?.profileImageURL}
              userLogin={broadcaster?.login}
            />
          }
          links={
            <>
              {broadcaster && (
                <CoreText
                  children={getFormattedUserDisplayName(broadcaster)}
                  color={Color.Base}
                  ellipsis
                />
              )}
              {stream.game?.displayName ? (
                <CoreText
                  children={stream.game.displayName}
                  color={Color.Base}
                  ellipsis
                />
              ) : null}
            </>
          }
          title={broadcaster?.broadcastSettings?.title ?? ''}
        />
      }
    />
  );
};

StreamCardBase.displayName = 'StreamCardBase';

export type FocusableStreamCardProps = Pick<
  FocusableCardLinkProps,
  'focusIndex' | 'marginBottom' | 'onClick' | 'onFocus'
> & {
  stream: FocusableStreamCard_stream;
};

// istanbul ignore next: trivial
export const FocusableStreamCardBase: FC<FocusableStreamCardProps> = ({
  focusIndex,
  marginBottom,
  onClick,
  onFocus,
  stream,
}) => {
  const { interactionContent } = useTrackedListItem(focusIndex);

  const renderLink = stream.broadcaster?.login
    ? renderTachyonLink({
        interactionContent,
        route: RouteName.Channel,
        routeParams: { login: stream.broadcaster.login },
      })
    : renderTachyonLink({
        interactionContent,
        route: RouteName.NotFound,
      });

  return (
    <FocusableCardLink
      config={WATCHABLE_CARD_CONFIG}
      focusIndex={focusIndex}
      marginBottom={marginBottom}
      onClick={onClick}
      onFocus={onFocus}
      renderLink={renderLink}
    >
      <StreamCardBase focusIndex={focusIndex} stream={stream} />
    </FocusableCardLink>
  );
};

FocusableStreamCardBase.displayName = 'FocusableStreamCardBase';

export const FocusableStreamCard = createFragmentContainer(
  FocusableStreamCardBase,
  {
    stream: graphql`
      fragment FocusableStreamCard_stream on Stream {
        broadcaster {
          displayName
          login
          hosting {
            id
          }
          broadcastSettings {
            title
          }
          profileImageURL(width: 50)
        }
        game {
          displayName
          name
        }
        id
        previewImageURL
        type
        viewersCount
      }
    `,
  },
);

FocusableStreamCard.displayName = 'FocusableStreamCard';
