import type { FC, FormEvent, KeyboardEvent } from 'react';
import { useEffect, useState } from 'react';
import styled from 'styled-components';
import { refFocusHandler, useFocus } from 'tachyon-tv-nav';
import { legacyKeyboardEventHandler } from 'tachyon-utils';
import type { InputProps } from 'twitch-core-ui';
import { Input, InputSize } from 'twitch-core-ui';
import { focusableBorderStyle } from '../styleMixins';

type FocusableTextInputProps = Omit<
  InputProps,
  'onChange' | 'onKeyDown' | 'onSubmit' | 'ref' | 'size'
> & {
  focusIndex: number;
  onSubmit: (inputValue: string) => void;
};

export const ScTextInputWrapper = styled.button<{ $focused: boolean }>`
  position: relative;
  width: 100%;

  &::after {
    content: '';
    height: calc(100% - 0.25rem);
    left: 0;
    ${focusableBorderStyle}
    position: absolute;
    top: 0;
    width: calc(100% - 0.5rem);
  }
`;

/**
 * Provides a Focusable Text Input. Prevents the underlying input from being focused
 * without an additional enter press to confirm user intent. This prevents the
 * native keyboard from unexpectedly taking focus which is hard to get out of.
 */
export const FocusableTextInput: FC<FocusableTextInputProps> = ({
  focusIndex,
  onSubmit,
  ...inputProps
}) => {
  const { focused, takeFocus } = useFocus(focusIndex);
  const [inputVal, setInputVal] = useState('');
  const [inputIsToggled, setInputIsToggled] = useState(false);

  function handleChange(event: FormEvent<HTMLInputElement>): void {
    setInputVal(event.currentTarget.value);
  }

  function handleKeyDown(event: KeyboardEvent<HTMLInputElement>): void {
    if (inputVal && event.key === 'Enter') {
      onSubmit(inputVal);
    }
  }

  useEffect(() => {
    if (!focused) {
      return;
    }

    const onKeydownHandler = legacyKeyboardEventHandler((event) => {
      if (event.key === 'Enter') {
        setInputIsToggled(true);
      }
    });

    const onClickHandler = () => {
      setInputIsToggled(true);
    };

    document.addEventListener('keydown', onKeydownHandler);
    document.addEventListener('click', onClickHandler);
    return () => {
      setInputIsToggled(false);
      document.removeEventListener('keydown', onKeydownHandler);
      document.removeEventListener('click', onClickHandler);
    };
  }, [focused]);

  const wrapperIsFocused = focused && !inputIsToggled;
  const inputIsFocused = focused && inputIsToggled;

  return (
    // The input wrapper exists to so that we can highlight the input on page load without automatically triggering
    // the system keyboard (the mode is annoying to exit) until the user confirms that they want to interact with the input
    // rather than use a previous search term.
    <ScTextInputWrapper
      $focused={wrapperIsFocused}
      onMouseEnter={takeFocus}
      ref={refFocusHandler(wrapperIsFocused)}
    >
      <Input
        onChange={handleChange}
        onKeyDown={handleKeyDown}
        ref={refFocusHandler(inputIsFocused)}
        size={InputSize.Large}
        value={inputVal}
        {...inputProps}
      />
    </ScTextInputWrapper>
  );
};

FocusableTextInput.displayName = 'FocusableTextInput';
