import type { FC } from 'react';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { RemSVG, squareSize } from 'tachyon-more-ui';
import { useUserIntent } from 'tachyon-user-intent';
import { SVGAsset } from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import { CARD_HEIGHT_REM, OVERSCAN_PADDING_REM } from '../../../../config';
import { useWandActivity } from '../../../framework';
import { fadeIn } from '../../styleMixins';

// Needs to fully cover the overscan area plus a little bit more to ensure
// that offscreen cards can't accidentally be hovered / focused when trying to
// press the button to scroll the shelf
const BUTTON_WIDTH_REM = OVERSCAN_PADDING_REM;

interface ScPageButtonProps {
  nextPage?: true;
}

// instabul ignore next
function getVariableAlignment({ nextPage }: ScPageButtonProps): string {
  if (nextPage) {
    return `
      right: 0;
    `;
  }

  return `
    left: 0;
    justify-content: flex-end;
  `;
}

// instabul ignore next
export const ScPageButton = styled.button<ScPageButtonProps>`
  align-items: center;
  display: flex !important;
  height: ${CARD_HEIGHT_REM}rem;
  ${getVariableAlignment}
  ${fadeIn({ $durationSeconds: 0.25 })}
  position: absolute;
  width: ${BUTTON_WIDTH_REM}rem;

  &:focus {
    outline: none;
  }
`;

// instabul ignore next
const ScSvgWrapper = styled.div`
  align-items: center;
  background: ${DarkThemeMap['color-background-button-primary-default']};
  border-radius: 50%;
  display: flex;
  justify-content: center;
  ${squareSize({ sizeRem: BUTTON_WIDTH_REM })}
  transition: all 0.1s linear;

  ${ScPageButton}:disabled & {
    background: ${DarkThemeMap['color-background-button-disabled']};
    transform: scale3d(0, 0, 1);
  }

  ${ScPageButton}:enabled:hover & {
    background: ${DarkThemeMap['color-background-button-primary-hover']};
    border-radius: 0;
    height: ${CARD_HEIGHT_REM}rem;
    width: ${BUTTON_WIDTH_REM}rem;
  }

  ${ScPageButton}:enabled:active & {
    background: ${DarkThemeMap['color-background-button-primary-active']};
    transform: scale3d(1.1, 1.1, 1);
  }
`;

type ShelfPageButtonsProps = {
  next: boolean;
  prev: boolean;
};

export const ShelfPageButtons: FC<ShelfPageButtonsProps> = ({ next, prev }) => {
  const { triggerIntent } = useUserIntent();
  const { formatMessage } = useIntl();
  const wandIsActive = useWandActivity();

  if (!wandIsActive) {
    return null;
  }

  return (
    <>
      <ScPageButton
        aria-label={formatMessage('Left arrow', 'ShelfPageButtons')}
        disabled={!prev}
        onClick={(e) => {
          e.currentTarget.blur();
          triggerIntent('page-left');
        }}
      >
        <ScSvgWrapper>
          <RemSVG
            asset={SVGAsset.GlyphArrLeft}
            heightRem={BUTTON_WIDTH_REM}
            widthRem={BUTTON_WIDTH_REM}
          />
        </ScSvgWrapper>
      </ScPageButton>
      <ScPageButton
        aria-label={formatMessage('Right arrow', 'ShelfPageButtons')}
        disabled={!next}
        nextPage
        onClick={(e) => {
          e.currentTarget.blur();
          triggerIntent('page-right');
        }}
      >
        <ScSvgWrapper>
          <RemSVG
            asset={SVGAsset.GlyphArrRight}
            heightRem={BUTTON_WIDTH_REM}
            widthRem={BUTTON_WIDTH_REM}
          />
        </ScSvgWrapper>
      </ScPageButton>
    </>
  );
};

ShelfPageButtons.displayName = 'ShelfPageButtons';
