import { useCallback } from 'react';
import type { HorizontalNavProps } from 'tachyon-tv-nav';
import { HorizontalNav, useFocus } from 'tachyon-tv-nav';
import type { CardConfig } from '../../../config';
import { CARD_ANIMATION_TIMING_S } from '../../../config';
import type { MovableListProps, VisibleRange } from '../MovableList';
import { MovableList } from '../MovableList';
import type { ShelfFrameProps } from './ShelfFrame';
import { ShelfFrame } from './ShelfFrame';
import { ShelfPageButtons } from './ShelfPageButtons';

export { SHELF_FRAME_HEIGHT_REM } from './ShelfFrame';

export type HorizontalShelfProps<T> = Pick<
  MovableListProps<T>,
  'itemRenderer' | 'items'
> &
  Pick<HorizontalNavProps, 'takeFocusOnFirstRender'> &
  ShelfFrameProps & {
    config: CardConfig;
    focusIndex: number;
    /**
     * Whether this shelf is nested inside a parent virtualized list
     */
    virtualized?: boolean;
  };

export function HorizontalShelf<T>({
  config,
  focusIndex,
  itemRenderer,
  items,
  takeFocusOnFirstRender,
  title,
  virtualized,
}: HorizontalShelfProps<T>): JSX.Element {
  const leadingBufferLines = 1;
  const trailingBufferLines = 1;
  const { focused } = useFocus(focusIndex);

  const shelfButtons = useCallback(
    (range: VisibleRange) =>
      focused && (
        <ShelfPageButtons
          next={
            range[range.length - 1] < items.length - 1 + trailingBufferLines
          }
          prev={range[0] > 0 - leadingBufferLines}
        />
      ),
    [focused, items.length],
  );

  return (
    <ShelfFrame title={title}>
      <HorizontalNav
        elementCount={items.length}
        focusIndex={focusIndex}
        pageSize={config.cardsPerRow}
        takeFocusOnFirstRender={takeFocusOnFirstRender}
        virtualizable={virtualized}
      >
        <MovableList
          direction="horizontal"
          elementsPerLine={1}
          itemRenderer={itemRenderer}
          items={items}
          leadingBufferLines={leadingBufferLines}
          lineSizeRem={config.widthRem}
          overlayRenderer={shelfButtons}
          trailingBufferLines={trailingBufferLines}
          transitionDurationS={CARD_ANIMATION_TIMING_S}
          visibleLines={config.cardsPerRow}
        />
      </HorizontalNav>
    </ShelfFrame>
  );
}

HorizontalShelf.displayName = 'HorizontalShelf';
