import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { StreamType, formatFollowers } from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AlignItems,
  Color,
  CoreText,
  Display,
  FlexDirection,
  Layout,
  Title,
  TitleSize,
  WhiteSpace,
} from 'twitch-core-ui';
import { GradientAvatar } from '../../GradientAvatar';
import type { BroadcasterOverview_channel$key } from './__generated__/BroadcasterOverview_channel.graphql';

// istanbul ignore next: trivial
const ScChannelAvatar = styled(Layout)`
  margin-right: 1.7rem;
`;

// istanbul ignore next: trivial
const broadcasterOverviewFragment = graphql`
  fragment BroadcasterOverview_channel on User {
    login
    displayName
    followers {
      totalCount
    }
    primaryColorHex
    primaryTeam {
      displayName
    }
    profileImageURL(width: 70)
  }
`;

export type BroadcasterOverviewProps = {
  channel: BroadcasterOverview_channel$key | null;
  isChannelLive: boolean;
};

/**
 * Component used for consistent layout across interstitial pages (channel, vod, & clips).
 */
export const BroadcasterOverview: FC<BroadcasterOverviewProps> = ({
  channel: channelRef,
  isChannelLive,
}) => {
  const channel = useFragment(broadcasterOverviewFragment, channelRef);
  const intl = useIntl();
  const { formatMessage } = intl;
  if (!channel) {
    return null;
  }
  const name = getFormattedUserDisplayName(channel);

  const followerCount = channel.followers?.totalCount;
  const teamName = channel.primaryTeam?.displayName;
  return (
    <Layout
      alignItems={AlignItems.Center}
      display={Display.Flex}
      flexDirection={FlexDirection.Row}
    >
      <ScChannelAvatar>
        <GradientAvatar
          login={channel.login}
          profileImageURL={channel.profileImageURL}
          streamType={isChannelLive ? StreamType.Live : undefined}
          streamerPrimaryColorHex={channel.primaryColorHex}
        />
      </ScChannelAvatar>
      <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
        <Title size={TitleSize.Small}>{name}</Title>
        <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
          {followerCount && (
            <CoreText color={Color.Base} ellipsis>
              {formatFollowers(intl, followerCount)}
            </CoreText>
          )}
          {followerCount && teamName && (
            <CoreText color={Color.Alt2} whiteSpace={WhiteSpace.PreWrap}>
              {' | '}
            </CoreText>
          )}
          {teamName && (
            <CoreText>
              {formatMessage(
                'Team: {teamName}',
                {
                  teamName,
                },
                'BroadcasterOverview',
              )}
            </CoreText>
          )}
        </Layout>
      </Layout>
    </Layout>
  );
};

BroadcasterOverview.displayName = 'InterstitialLayoutBroadcasterOverview';
