import { image, internet } from 'faker';
import { useFragment } from 'react-relay/hooks';
import { formatFollowers } from 'tachyon-more-ui';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { BroadcasterOverview } from '.';

jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useFragment: jest.fn(),
}));
const mockUseFragment = useFragment as jest.Mock;

jest.mock('tachyon-more-ui', () => ({
  ...jest.requireActual('tachyon-more-ui'),
  formatFollowers: jest.fn(),
}));

describe(BroadcasterOverview, () => {
  beforeEach(() => {
    mockUseFragment.mockReset();
  });

  const setup = createShallowWrapperFactory(BroadcasterOverview, () => ({
    channel: {} as any,
    isChannelLive: true,
  }));

  const TEAM_NAME = internet.userName();

  const mockUser = {
    ' $refType': 'channel',
    displayName: internet.userName(),
    followers: {
      totalCount: 12,
    },
    primaryColorHex: '',
    primaryTeam: {
      displayName: TEAM_NAME,
    },
    profileImageURL: image.imageUrl(),
  };

  it('does not render a followerCount if none exists', () => {
    mockUseFragment.mockImplementationOnce(() => ({
      ...mockUser,
      followers: undefined,
    }));
    setup();
    expect(formatFollowers).toHaveBeenCalledTimes(0);
  });

  it('renders a followerCount if one exists', () => {
    mockUseFragment.mockImplementationOnce(() => mockUser);
    setup();
    expect(formatFollowers).toHaveBeenCalledTimes(1);
  });

  it('renders a team name if one exists', () => {
    mockUseFragment.mockImplementationOnce(() => mockUser);
    const { wrapper } = setup();
    expect(wrapper.find({ children: `Team: ${TEAM_NAME}` })).toExist();
  });

  it('renders a spacer if team and follower count exist', () => {
    mockUseFragment.mockImplementationOnce(() => mockUser);
    const { wrapper } = setup();
    expect(wrapper.find({ children: ' | ' })).toExist();
  });

  it('does not render a spacer if only one of team and follower count exists', () => {
    mockUseFragment.mockImplementationOnce(() => ({
      ...mockUser,
      primaryTeam: { displayName: null },
    }));
    const { wrapper } = setup();
    expect(wrapper.find({ children: ' | ' })).not.toExist();
  });
});
