import { usePlaybackState } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import styled from 'styled-components';
import { useCurrentUser } from 'tachyon-auth';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { HorizontalNav } from 'tachyon-tv-nav';
import { usePrevious } from 'tachyon-utils';
import { Display, Layout, SVGAsset } from 'twitch-core-ui';
import type { IsContentPlayable } from '../../../../hooks';
import {
  RouteName,
  createRedirectQuery,
  renderTachyonLink,
  routeNameFromPathname,
  useTachyonRouter,
} from '../../../../routing';
import { FlyOutMenu } from '../../FlyOutMenu';
import { FocusableFollowButton } from '../../FocusableFollowButton';
import { FocusableReportButton } from '../../FocusableReportButton';
import { FocusableTextButton } from '../../FocusableTextButton';
import type { InterstitialButtonRow_channel$key } from './__generated__/InterstitialButtonRow_channel.graphql';

// istanbul ignore next: trivial
const ScFocusableButton = styled.div`
  margin-right: 1rem;
`;

export type InterstitialContentType = 'clip' | 'stream' | 'vod';

export type ButtonRowProps = {
  channel: InterstitialButtonRow_channel$key | null;
  contentType: InterstitialContentType;
  focusIndex: number;
  fullVideoID?: string;
  isContentPlayable: IsContentPlayable;
  triggerPlayAttempt: () => void;
};

// istanbul ignore next: trivial
const interstitialButtonRowFragment = graphql`
  fragment InterstitialButtonRow_channel on User {
    ...FocusableFollowButton_channel
    login
  }
`;

// istanbul ignore next: trivial
export const InterstitialButtonRow: FC<ButtonRowProps> = ({
  channel: channelRef,
  contentType,
  focusIndex,
  fullVideoID,
  isContentPlayable,
  triggerPlayAttempt,
}) => {
  const { currentPathname } = useRouterUtils();
  const { push } = useTachyonRouter();
  const { loggedIn } = useCurrentUser();
  const channel = useFragment(interstitialButtonRowFragment, channelRef);
  const { formatMessage } = useIntl();
  useExitedStreamInNativePlayerTracker(contentType);

  let buttonText = '';
  switch (contentType) {
    case 'stream':
      buttonText = formatMessage('Watch Live', 'InterstitialButtonRow');
      break;
    case 'vod':
      buttonText = formatMessage('Watch VOD', 'InterstitialButtonRow');
      break;
    case 'clip':
      buttonText = formatMessage('Watch Clip', 'InterstitialButtonRow');
      break;
  }

  const flyOutButtons = [];
  const buttons = [
    <ScFocusableButton key="play">
      <FocusableTextButton
        focusIndex={0}
        icon={SVGAsset.Play}
        onClick={async () => {
          // isContentPlayable relies on a child component providing a value
          // higher up in the tree via a callback so we delay calling the
          // function until we need to. In this case, the user pressing
          // the button.
          if (await isContentPlayable()) {
            triggerPlayAttempt();
          } else {
            push({ route: RouteName.WrongFormat });
          }
        }}
        withUnfocusBackground
      >
        {buttonText}
      </FocusableTextButton>
    </ScFocusableButton>,
  ];

  if (channel) {
    buttons.push(
      <ScFocusableButton key="follow">
        <FocusableFollowButton
          channel={channel}
          focusIndex={1}
          route={
            contentType === 'clip'
              ? RouteName.ChannelHome
              : routeNameFromPathname(currentPathname)
          }
          withUnfocusBackground
        />
      </ScFocusableButton>,
      <ScFocusableButton key="channelHome">
        <FocusableTextButton
          focusIndex={2}
          icon={SVGAsset.Account}
          linkTo="/deferToRenderLink"
          renderLink={renderTachyonLink({
            route: RouteName.ChannelHome,
            routeParams: { login: channel.login },
          })}
          withUnfocusBackground
        >
          {formatMessage('Go to Profile', 'InterstitialButtonRow')}
        </FocusableTextButton>
      </ScFocusableButton>,
    );
    flyOutButtons.push(
      <FocusableTextButton
        contentLeft
        focusIndex={0}
        fullWidth
        icon={SVGAsset.NavCreative}
        key="not-interested"
        linkTo="/deferToRenderLink"
        renderLink={
          loggedIn
            ? renderTachyonLink({
                route: RouteName.SettingsChannelNotInterested,
                routeParams: {
                  login: channel.login,
                },
              })
            : renderTachyonLink({
                query: createRedirectQuery({
                  route: RouteName.SettingsChannelNotInterested,
                  routeParams: { login: channel.login },
                }),
                route: RouteName.Login,
              })
        }
      >
        {formatMessage('Not Interested', 'InterstitialButtonRow')}
      </FocusableTextButton>,
      <FocusableReportButton
        contentLeft
        focusIndex={1}
        fullWidth
        key="report"
        login={channel.login}
      />,
    );
  }

  if (fullVideoID) {
    flyOutButtons.push(
      <FocusableTextButton
        contentLeft
        focusIndex={flyOutButtons.length}
        fullWidth
        icon={SVGAsset.NavVideos}
        linkTo="/deferToRenderLink"
        renderLink={renderTachyonLink({
          route: RouteName.VodInterstitial,
          routeParams: { id: fullVideoID },
        })}
      >
        {formatMessage('Watch Full Video', 'InterstitialButtonRow')}
      </FocusableTextButton>,
    );
  }

  const showFlyOutButton = flyOutButtons.length > 0;

  return (
    <HorizontalNav
      elementCount={buttons.length + (showFlyOutButton ? 1 : 0)}
      focusIndex={focusIndex}
    >
      <Layout display={Display.Flex} padding={{ top: 2 }}>
        {buttons}
        {showFlyOutButton && (
          <FlyOutMenu
            aria-label={formatMessage('More Options', 'InterstitialButtonRow')}
            elementCount={flyOutButtons.length}
            focusIndex={buttons.length}
            icon={SVGAsset.More}
            initialChildFocusIndex={0}
            withUnfocusBackground
          >
            {flyOutButtons}
          </FlyOutMenu>
        )}
      </Layout>
    </HorizontalNav>
  );
};

InterstitialButtonRow.displayName = 'InterstitialButtonRow';

/**
 * tracking hook for switch when a user returns to the interstitial
 * page from the native player. Observing the playbackstate is the
 * simplest way to identify this.
 */
export function useExitedStreamInNativePlayerTracker(
  contentType: InterstitialContentType,
): void {
  const trackInteraction = useInteractionTracking();
  const playerState = usePlaybackState();
  const previousPlayerState = usePrevious(playerState);

  if (
    contentType === 'stream' &&
    previousPlayerState === 'Playing' &&
    playerState === 'Idle'
  ) {
    trackInteraction({
      interaction: InteractionType.Impression,
      interactionContent: 'exited_stream_in_native_player',
    });
  }
}
