import { act, renderHook } from '@testing-library/react-hooks';
import {
  mockPlayerController,
  usePlaybackState,
  usePlayerController,
} from 'pulsar';
import { useFragment } from 'react-relay/hooks';
import { useInteractionTracking } from 'tachyon-event-tracker';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { RouteName } from '../../../../routing';
import { FocusableTextButton } from '../../FocusableTextButton';
import { InterstitialButtonRow, useExitedStreamInNativePlayerTracker } from '.';

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlaybackState: jest.fn(),
  usePlayerController: jest.fn(() => null),
}));

jest.mock('tachyon-event-tracker', () => ({
  ...jest.requireActual('tachyon-event-tracker'),
  useInteractionTracking: jest.fn(),
}));

jest.mock('react-relay/hooks', () => ({
  ...jest.requireActual('react-relay/hooks'),
  useFragment: jest.fn(),
}));

const mockRouterPush = jest.fn();

jest.mock('../../../../routing', () => ({
  ...jest.requireActual('../../../../routing'),
  useTachyonRouter: () => ({
    push: mockRouterPush,
  }),
}));

const mockUseFragment = useFragment as jest.Mock;
const mockUsePlaybackState = usePlaybackState as jest.Mock;
const mockUsePlayerController = usePlayerController as jest.Mock;
const mockUseInteractionTracking = useInteractionTracking as jest.Mock;

const mockChannel = null;

describe(useExitedStreamInNativePlayerTracker, () => {
  const setup = createShallowWrapperFactory(InterstitialButtonRow, () => ({
    channel: {} as any,
    contentType: 'stream',
    focusIndex: 0,
    isContentPlayable: () => Promise.resolve(true),
    triggerPlayAttempt: () => undefined,
  }));

  beforeEach(() => {
    mockUseFragment.mockImplementation(() => mockChannel);
  });

  const mockTrackInteraction = jest.fn();
  mockUseInteractionTracking.mockImplementation(() => mockTrackInteraction);

  it('fires a tracking event after the player has played and then been paused', () => {
    mockUsePlaybackState.mockReturnValueOnce('Idle');
    const { rerender } = renderHook(() => {
      useExitedStreamInNativePlayerTracker('stream');
    });

    expect(mockTrackInteraction).not.toHaveBeenCalled();

    mockUsePlaybackState.mockReturnValueOnce('Playing');
    rerender();

    expect(mockTrackInteraction).not.toHaveBeenCalled();

    mockUsePlaybackState.mockReturnValueOnce('Idle');
    rerender();

    expect(mockTrackInteraction).toHaveBeenCalledTimes(1);
  });

  it('redirects when the content is not supported', async () => {
    const mockPlay = jest.fn();
    const mockedPlayerController = mockPlayerController({
      play: mockPlay,
    });
    mockUsePlayerController.mockReturnValue(mockedPlayerController);

    const { wrapper } = setup({
      isContentPlayable: () => Promise.resolve(false),
    });

    await act(async () => {
      wrapper.find(FocusableTextButton).props().onClick!({
        stopPropagation: jest.fn(),
      } as any);

      // Allow the promise inside of the onClick to resolve
      await Promise.resolve();
    });

    expect(mockPlay).not.toHaveBeenCalled();
    expect(mockRouterPush).toHaveBeenCalledWith({
      route: RouteName.WrongFormat,
    });
  });

  it('initiates playback when the content is supported', async () => {
    const { props, wrapper } = setup({
      triggerPlayAttempt: jest.fn(),
    });
    await act(async () => {
      wrapper.find(FocusableTextButton).props().onClick!({
        stopPropagation: jest.fn(),
      } as any);

      // Allow the promise inside of the onClick to resolve
      await Promise.resolve();
    });

    expect(props.triggerPlayAttempt).toHaveBeenCalled();
  });
});
