import { usePlaybackRestrictions, usePlayerController } from 'pulsar';
import type { FC } from 'react';
import { useEffect, useMemo } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import styled from 'styled-components';
import { HorizontalNav, NodeNav } from 'tachyon-tv-nav';
import {
  Background,
  BorderRadius,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  Overflow,
} from 'twitch-core-ui';
import { OVERSCAN_PADDING_REM, TOP_NAV_HEIGHT_REM } from '../../../config';
import type { IsContentPlayable } from '../../../hooks';
import { BroadcasterOverview } from './BroadcasterOverview';
import { ChannelDescription } from './ChannelDescription';
import { InterstitialButtonRow } from './InterstitialButtonRow';
import type { InterstitialContentType } from './InterstitialButtonRow';
import type { InterstitialLayout_channel$key } from './__generated__/InterstitialLayout_channel.graphql';

export { InterstitialVideoDetails } from './InterstitialVideoDetails';

// istanbul ignore next: trivial
const ScInterstitialLayoutContainer = styled(Layout)`
  padding: ${TOP_NAV_HEIGHT_REM + 2}rem ${OVERSCAN_PADDING_REM}rem
    ${OVERSCAN_PADDING_REM}rem;
`;

// istanbul ignore next: trivial
const ScChannelDetails = styled.div`
  padding-left: 8rem;
  padding-top: 1rem;
`;

// istanbul ignore next: trivial
const interstitialLayoutFragment = graphql`
  fragment InterstitialLayout_channel on User {
    ...BroadcasterOverview_channel
    ...ChannelDescription_channel
    ...InterstitialButtonRow_channel
  }
`;

export type InterstitialLayoutProps = {
  attemptedPlay: boolean;
  channel: InterstitialLayout_channel$key | null;
  /**
   * JSX Element for the channel's chat pane. Set focusIndex
   * to 0 to ensure navigation works properly. Can be omitted
   * if no chat pane is desired.
   */
  chat?: JSX.Element | null;
  contentType: InterstitialContentType;
  /**
   * Only used for clips so we can link to the full vod.
   */
  fullVideoID?: string;
  isChannelLive: boolean;
  isContentPlayable: IsContentPlayable;
  player: JSX.Element;
  resetAttemptedPlay: () => void;
  triggerPlayAttempt: () => void;
};

// istanbul ignore next: trivial
/**
 * Component used for consistent presentation of interstitial layouts on Switch.
 */
export const InterstitialLayout: FC<InterstitialLayoutProps> = ({
  attemptedPlay,
  channel: channelRef,
  chat,
  children,
  contentType,
  fullVideoID,
  isChannelLive,
  isContentPlayable,
  player,
  resetAttemptedPlay,
  triggerPlayAttempt,
}) => {
  const channel = useFragment(interstitialLayoutFragment, channelRef);
  const controller = usePlayerController();
  const { activeRestriction } = usePlaybackRestrictions();

  useEffect(() => {
    if (activeRestriction === null && attemptedPlay && controller) {
      controller.play();
      resetAttemptedPlay();
    }
  }, [activeRestriction, attemptedPlay, controller, resetAttemptedPlay]);

  // We memoize the return value because the subtree is relatively expensive
  // and will get thrashed by the usePlaybackState hook which is consumed
  // in restriction components that wrap this one.
  return useMemo(
    () => (
      <>
        <HorizontalNav
          elementCount={chat ? 2 : 1}
          focusIndex={0}
          takeFocusOnFirstRender
        >
          <ScInterstitialLayoutContainer display={Display.Flex} fullHeight>
            <Layout
              display={Display.Flex}
              flexDirection={FlexDirection.Column}
              flexShrink={0}
              justifyContent={JustifyContent.End}
              padding={{ bottom: 2, right: 4 }}
              width="74%"
            >
              <BroadcasterOverview
                channel={channel}
                isChannelLive={isChannelLive}
              />
              <ScChannelDetails>
                {children}
                <ChannelDescription channel={channel} />
                <InterstitialButtonRow
                  channel={channel}
                  contentType={contentType}
                  focusIndex={0}
                  fullVideoID={fullVideoID}
                  isContentPlayable={isContentPlayable}
                  triggerPlayAttempt={triggerPlayAttempt}
                />
              </ScChannelDetails>
            </Layout>
            {chat && (
              <NodeNav focusIndex={1}>
                <Layout
                  background={Background.Base}
                  borderRadius={BorderRadius.Medium}
                  fullHeight
                  fullWidth
                  overflow={Overflow.Hidden}
                >
                  {chat}
                </Layout>
              </NodeNav>
            )}
          </ScInterstitialLayoutContainer>
        </HorizontalNav>
        <Layout display={Display.HideAccessible}>{player}</Layout>
      </>
    ),
    [
      chat,
      channel,
      children,
      contentType,
      fullVideoID,
      isChannelLive,
      isContentPlayable,
      player,
      triggerPlayAttempt,
    ],
  );
};

InterstitialLayout.displayName = 'InterstitialLayout';
