import styled from 'styled-components';
import type { CreateVisibleRangeOpts } from '../createVisibleRange';
import type { Direction } from '../types';

export type ScScrollContainerProps = Required<
  Omit<
    CreateVisibleRangeOpts,
    'elementsPerLine' | 'initialFocusIndex' | 'itemCount'
  >
> & {
  /**
   * Used for passing styles from the wrapping container.
   */
  className?: string;
  /**
   * Describes the type of list and how it scrolls and is layed out.
   * - horizontal: single row list that scrolls horizontally
   * - vertical: single column list that scrolls vertically
   * - vertical-grid: multi-column list that scrolls vertically
   */
  direction: Direction;
  /**
   * The size to offset with each step. For vertical lists, this should be the
   * height of a child element. For horizontal lists, this should be the width
   * of a child element. This value should include any padding/margin, which the
   * child should provide itself (this component does not provide intra-child
   * spacing).
   */
  lineSizeRem: number;
  /**
   * The number of lines we have scrolled into the list UI.
   */
  offsetLineCount: number;
  /**
   * The total number of lines that the list occupies.
   */
  totalLines: number;
  /**
   * The duration (in seconds) for the transistion animations.
   */
  transitionDurationS: number;
  /**
   * The width of a container for vertical lists.
   */
  verticalWidthVw?: number;
};

// istanbul ignore next: trivial
// `position: fixed` usage defeats Chrome's scroll-to-focused behavior which
// messes up our animations
export const ScScrollContainer = styled.div<ScScrollContainerProps>`
  ${getLayoutStyles}
  ${getSpacingStyles}
  position: fixed;
  transition: transform
    ${(props: ScScrollContainerProps): number => props.transitionDurationS}s
    ease-in-out;
`;

type SpacingSizes = {
  marginRem: number;
  paddingRem: number;
};

export function getSpacingSizes({
  leadingBufferLines,
  lineSizeRem,
  offsetLineCount,
}: ScScrollContainerProps): SpacingSizes {
  return {
    marginRem: offsetLineCount * lineSizeRem,
    paddingRem: Math.max(0, offsetLineCount - leadingBufferLines) * lineSizeRem,
  };
}

function getSpacingStyles(props: ScScrollContainerProps): string {
  const { marginRem, paddingRem } = getSpacingSizes(props);

  if (props.direction === 'horizontal') {
    return `
      padding-left: ${paddingRem}rem;
      transform: translate3d(-${marginRem}rem, 0, 0);
    `;
  }

  return `
    padding-top: ${paddingRem}rem;
    transform: translate3d(0, -${marginRem}rem, 0);
  `;
}

// istanbul ignore next: trivial
export function getHorizontalWidth({
  lineSizeRem,
  totalLines,
}: ScScrollContainerProps): string {
  return `width: ${totalLines * lineSizeRem}rem;`;
}

// istanbul ignore next: trivial
export function getVerticalWidth({
  verticalWidthVw,
}: ScScrollContainerProps): string {
  if (!verticalWidthVw) {
    return '';
  }

  return `width: ${verticalWidthVw}vw;`;
}

function getLayoutStyles(props: ScScrollContainerProps): string {
  switch (props.direction) {
    case 'horizontal':
      return `
        display: flex;
        ${getHorizontalWidth(props)}
      `;
    case 'vertical':
      return `
        display: flex;
        flex-direction: column;
        ${getVerticalWidth(props)}
      `;
    case 'vertical-grid':
      return `
        align-content: start;
        display: flex;
        flex-wrap: wrap;
        justify-content: start;
        ${getVerticalWidth(props)}
      `;
    default:
      const exhaust: never = props.direction;
      return exhaust;
  }
}
