import type { VisibleRange, VisibleRangeBaseOpts } from '../types';

export interface CreateVisibleRangeOpts extends VisibleRangeBaseOpts {
  /**
   * The index of the initially focused items. Defaults to 0.
   */
  initialFocusIndex?: number;
  /**
   * The number of items in the list.
   */
  itemCount: number;
  /**
   * The number of lines presented to the user for interaction (not including
   * clipped edge lines).
   */
  visibleLines: number;
}

export function createVisibleRange({
  elementsPerLine,
  initialFocusIndex = 0,
  itemCount,
  leadingBufferLines = 0,
  trailingBufferLines = 0,
  visibleLines,
}: CreateVisibleRangeOpts): VisibleRange {
  const focusedLine = Math.floor(initialFocusIndex / elementsPerLine);
  // use visibleLines as a way to create "pages" of visible elements to
  // preserve a sense of page location when recreating a list
  const focusedLineClampedToPage =
    Math.floor(focusedLine / visibleLines) * visibleLines;

  const range = Array.from(
    Array(
      (visibleLines + leadingBufferLines + trailingBufferLines) *
        elementsPerLine,
    ).keys(),
  ).map(
    (n) =>
      n + (focusedLineClampedToPage - leadingBufferLines) * elementsPerLine,
  );

  // ensure that if we're on the last page, every line has content
  const lastIndex = range[range.length - 1];
  const lastPossibleIndex =
    (Math.ceil(itemCount / elementsPerLine) + trailingBufferLines) *
      elementsPerLine -
    1;
  if (lastIndex <= lastPossibleIndex) {
    return range;
  }

  return range.map((n) => n - (lastIndex - lastPossibleIndex));
}
