import { createVisibleRange } from '.';

describe(createVisibleRange, () => {
  // default convenience values
  const itemCount = 20;
  const leadingBufferLines = 1;
  const trailingBufferLines = 1;

  it('works on single entity', () => {
    expect(
      createVisibleRange({
        elementsPerLine: 1,
        itemCount,
        visibleLines: 1,
      }),
    ).toEqual([0]);
  });

  describe('without buffers or initial focus', () => {
    it('works on straight line', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          itemCount,
          visibleLines: 2,
        }),
      ).toEqual([
        // line 0
        0,
        // line 1
        1,
      ]);
    });

    it('works on grid', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 2,
          itemCount,
          visibleLines: 2,
        }),
      ).toEqual([
        // line 0, item 0
        0,
        // line 0, item 1
        1,
        // line 1, item 0
        2,
        // line 1, item 1
        3,
      ]);
    });
  });

  describe('with buffer', () => {
    it('works on straight line', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 2,
        }),
      ).toEqual([
        // leading buffer 0
        -1,
        // line 0
        0,
        // line 1
        1,
        // trailing buffer 0
        2,
      ]);
    });

    it('works on grid', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 2,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 2,
        }),
      ).toEqual([
        // leading buffer 0, item 0
        -2,
        // leading buffer 0, item 1
        -1,
        // line 0, item 0
        0,
        // line 0, item 1
        1,
        // line 1, item 0
        2,
        // line 1, item 1
        3,
        // trailing buffer 0, item 0
        4,
        // trailing buffer 0, item 1
        5,
      ]);
    });
  });

  describe('with initial focused line', () => {
    it('works on straight line', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          initialFocusIndex: 1,
          itemCount,
          visibleLines: 2,
        }),
      ).toEqual([
        // line 0, item 0 (page 0)
        0,
        // line 0, item 1 (page 0)
        1,
      ]);
    });

    it('works on grid', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 2,
          initialFocusIndex: 2,
          itemCount,
          visibleLines: 2,
        }),
      ).toEqual([
        // line 0, item 0 (page 0)
        0,
        // line 0, item 1 (page 0)
        1,
        // line 1, item 0 (page 0)
        2,
        // line 1, item 1 (page 0)
        3,
      ]);
    });

    it('works on large grid with jagged leading edge', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 4,
          initialFocusIndex: 5,
          itemCount,
          visibleLines: 2,
        }),
      ).toEqual([
        // line 0, item 0 (page 0)
        0,
        // line 0, item 1 (page 0)
        1,
        // line 0, item 2 (page 0)
        2,
        // line 0, item 3 (page 0)
        3,
        // line 1, item 0 (page 0)
        4,
        // line 1, item 1 (page 0)
        5,
        // line 1, item 2 (page 0)
        6,
        // line 1, item 3 (page 0)
        7,
      ]);
    });
  });

  describe('with initial focused line and buffer', () => {
    it('works on straight line', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          initialFocusIndex: 1,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 2,
        }),
      ).toEqual([
        // leading buffer 0
        -1,
        // line 0, item 0 (page 0)
        0,
        // line 1, item 0 (page 0)
        1,
        // trailing buffer 0
        2,
      ]);
    });

    it('works on grid', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 2,
          initialFocusIndex: 2,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 2,
        }),
      ).toEqual([
        // leading buffer 0, item 0
        -2,
        // leading buffer 0, item 1
        -1,
        // line 1, item 0 (page 0)
        0,
        // line 1, item 1 (page 0)
        1,
        // line 2, item 0 (page 0)
        2,
        // line 2, item 1 (page 0)
        3,
        // trailing buffer 0, item 0
        4,
        // trailing buffer 0, item 1
        5,
      ]);
    });
  });

  describe('mid-list page alignment', () => {
    it('properly clamps to page 1 for first item of page', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          initialFocusIndex: 4,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 4,
        }),
      ).toEqual([
        // leading buffer 0
        3,
        // line 4, item 0 (page 1)
        4,
        // line 5, item 0 (page 1)
        5,
        // line 6, item 0 (page 1)
        6,
        // line 7, item 0 (page 1)
        7,
        // trailing buffer 0
        8,
      ]);
    });

    it('properly clamps to page 1 for second item of page', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          initialFocusIndex: 5,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 4,
        }),
      ).toEqual([
        // leading buffer 0
        3,
        // line 4, item 0 (page 1)
        4,
        // line 5, item 0 (page 1)
        5,
        // line 6, item 0 (page 1)
        6,
        // line 7, item 0 (page 1)
        7,
        // trailing buffer 0
        8,
      ]);
    });

    it('properly clamps to page 2', () => {
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          initialFocusIndex: 10,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 4,
        }),
      ).toEqual([
        // leading buffer 0
        7,
        // line 8, item 0 (page 2)
        8,
        // line 9, item 0 (page 2)
        9,
        // line 10, item 0 (page 2)
        10,
        // line 11, item 0 (page 2)
        11,
        // trailing buffer 0
        12,
      ]);
    });
  });

  describe('final page handling', () => {
    it('properly clamps to end for straight line for last item', () => {
      // going by visibleLines, the final page would only be half full
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          initialFocusIndex: 19,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 8,
        }),
      ).toEqual([
        // leading buffer 0 & line 11, item 0 (normally page 2)
        11,
        // line 12, item 0 (normally page 2)
        12,
        // line 13, item 0 (normally page 2)
        13,
        // line 14, item 0 (normally page 2)
        14,
        // line 15, item 0 (normally page 2)
        15,
        // line 16, item 0 (normally page 3)
        16,
        // line 17, item 0 (normally page 3)
        17,
        // line 18, item 0 (normally page 3)
        18,
        // line 19, item 0 (normally page 3)
        19,
        // trailing buffer 0
        20,
      ]);
    });
    it('properly clamps to end for straight line for item on last non-full page', () => {
      // going by visibleLines, the final page would only be half full
      expect(
        createVisibleRange({
          elementsPerLine: 1,
          initialFocusIndex: 18,
          itemCount,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 8,
        }),
      ).toEqual([
        // leading buffer 0 & line 11, item 0 (normally page 2)
        11,
        // line 12, item 0 (normally page 2)
        12,
        // line 13, item 0 (normally page 2)
        13,
        // line 14, item 0 (normally page 2)
        14,
        // line 15, item 0 (normally page 2)
        15,
        // line 16, item 0 (normally page 3)
        16,
        // line 17, item 0 (normally page 3)
        17,
        // line 18, item 0 (normally page 3)
        18,
        // line 19, item 0 (normally page 3)
        19,
        // trailing buffer 0 (no item)
        20,
      ]);
    });

    it('properly clamps to end for grid', () => {
      // going by visibleLines, the final page would only be half full
      // the last line itself ends up partially full but we want to preserve the per-line allocation
      expect(
        createVisibleRange({
          elementsPerLine: 2,
          initialFocusIndex: 18,
          itemCount: 19,
          leadingBufferLines,
          trailingBufferLines,
          visibleLines: 4,
        }),
      ).toEqual([
        // leading buffer 0, item 0 & line 5, item 0 (normally page 1)
        10,
        // leading buffer 0, item 1 & line 5, item 1 (normally page 1)
        11,
        // line 6, item 0 (normally page 1)
        12,
        // line 6, item 1 (normally page 1)
        13,
        // line 7, item 0 (normally page 1)
        14,
        // line 7, item 1 (normally page 1)
        15,
        // line 8, item 0 (normally page 2)
        16,
        // line 8, item 1 (normally page 2)
        17,
        // line 9, item 0 (normally page 2)
        18,
        // line 9, item 1 (empty / does not correspond to an item)
        19,
        // trailing buffer 0, item 0
        20,
        // trailing buffer 0, item 0
        21,
      ]);
    });
  });
});
