import type { FC, ReactNode } from 'react';
import { useMemo, useRef } from 'react';
import { useAreaChildFocusIndex } from 'tachyon-tv-nav';
import type { ScScrollContainerProps } from './ScScrollContainer';
import { ScScrollContainer } from './ScScrollContainer';
import type { CreateVisibleRangeOpts } from './createVisibleRange';
import { createVisibleRange } from './createVisibleRange';
import { getOffsetLineCount } from './getOffsetLineCount';
import { shouldVisibleRangeChange } from './shouldVisibleRangeChange';
import type { VisibleRange } from './types';
import { updateVisibleRange } from './updateVisibleRange';

export type { VisibleRange } from './types';

export type MovableListProps<T = any> = Omit<
  CreateVisibleRangeOpts,
  'initialFocusIndex' | 'itemCount'
> &
  Pick<
    ScScrollContainerProps,
    'direction' | 'lineSizeRem' | 'transitionDurationS' | 'verticalWidthVw'
  > & {
    /**
     * Function that renders an item given the item data and its index in the
     * array.
     */
    itemRenderer: (item: T, idx: number) => ReactNode;
    /**
     * The array of items to be rendered.
     */
    items: Readonly<T[]>;
    /**
     * Render prop for an overlay on the virtualized list which receives the current visibleRange
     */
    overlayRenderer?: (range: VisibleRange) => ReactNode;
  };

/**
 * A component for imparting animated "scrolling" transitions between lines in
 * a virtualized list or grid of equally size child elements in a Navigation Area.
 * A line is the directionless unit in the direction of scroll: in a vertically-oriented
 * list it is equivalent to a row, and in a horizontally-oriented list it is
 * equivalent to a column.
 */
export const MovableList: FC<MovableListProps> = ({
  direction,
  elementsPerLine,
  itemRenderer,
  items,
  leadingBufferLines = 0,
  lineSizeRem,
  overlayRenderer,
  trailingBufferLines = 0,
  transitionDurationS,
  verticalWidthVw,
  visibleLines,
}) => {
  const focusedChildIndex = useAreaChildFocusIndex();
  const visibleRangeRef = useRef<VisibleRange>();

  if (!visibleRangeRef.current) {
    visibleRangeRef.current = createVisibleRange({
      elementsPerLine,
      initialFocusIndex: focusedChildIndex,
      itemCount: items.length,
      leadingBufferLines,
      trailingBufferLines,
      visibleLines,
    });
  } else if (
    shouldVisibleRangeChange({
      elementsPerLine,
      leadingBufferLines,
      newFocusIndex: focusedChildIndex,
      trailingBufferLines,
      visibleRange: visibleRangeRef.current,
    })
  ) {
    visibleRangeRef.current = updateVisibleRange({
      currentVisibleRange: visibleRangeRef.current,
      elementsPerLine,
      leadingBufferLines,
      newFocusIndex: focusedChildIndex,
      trailingBufferLines,
    });
  }

  const range = visibleRangeRef.current;
  const { overlay, visibleItems } = useMemo(() => {
    const _overlay = overlayRenderer?.(range);

    const _visibleItems = range.reduce<ReactNode[]>((acc, idx) => {
      if (0 <= idx && idx < items.length) {
        acc.push(itemRenderer(items[idx], idx));
      }
      return acc;
    }, []);

    return { overlay: _overlay, visibleItems: _visibleItems };
  }, [range, overlayRenderer, itemRenderer, items]);

  return (
    <>
      <ScScrollContainer
        direction={direction}
        leadingBufferLines={leadingBufferLines}
        lineSizeRem={lineSizeRem}
        offsetLineCount={getOffsetLineCount({
          currentVisibleRange: visibleRangeRef.current,
          elementsPerLine,
          leadingBufferLines,
        })}
        totalLines={Math.ceil(items.length / elementsPerLine)}
        trailingBufferLines={trailingBufferLines}
        transitionDurationS={transitionDurationS}
        verticalWidthVw={verticalWidthVw}
        visibleLines={visibleLines}
      >
        {visibleItems}
      </ScScrollContainer>
      {overlay}
    </>
  );
};

MovableList.displayName = 'MovableList';
