import { updateVisibleRange } from '.';

describe(updateVisibleRange, () => {
  describe('simple case', () => {
    it('works on straight line', () => {
      const elementsPerLine = 1;
      const range0 = [
        // line 0, focused
        0,
        // line 1
        1,
        // line 2
        2,
        // line 3
        3,
      ];
      const range1 = updateVisibleRange({
        currentVisibleRange: range0,
        elementsPerLine,
        newFocusIndex: 2,
      });
      // float within existing without changing
      expect(range1).toEqual([
        // line 0
        0,
        // line 1
        1,
        // line 2, focused
        2,
        // line 3
        3,
      ]);

      const range2 = updateVisibleRange({
        currentVisibleRange: range1,
        elementsPerLine,
        newFocusIndex: 4,
      });
      // move down a line
      expect(range2).toEqual([
        // line 1
        1,
        // line 2
        2,
        // line 3
        3,
        // line 4, focused
        4,
      ]);

      const range3 = updateVisibleRange({
        currentVisibleRange: range2,
        elementsPerLine,
        newFocusIndex: 0,
      });
      // move up a line
      expect(range3).toEqual([
        // line 0, focused
        0,
        // line 1
        1,
        // line 2
        2,
        // line 3
        3,
      ]);
    });

    it('works on grid', () => {
      const elementsPerLine = 2;
      const range0 = [
        // line 0, item 0, focused
        0,
        // line 0, item 1
        1,
        // line 1, item 0
        2,
        // line 1, item 1
        3,
      ];
      const range1 = updateVisibleRange({
        currentVisibleRange: range0,
        elementsPerLine,
        newFocusIndex: 2,
      });
      // float within existing without changing
      expect(range1).toEqual([
        // line 0, item 0
        0,
        // line 0, item 1
        1,
        // line 1, item 0, focused
        2,
        // line 1, item 1
        3,
      ]);

      const range2 = updateVisibleRange({
        currentVisibleRange: range1,
        elementsPerLine,
        newFocusIndex: 5,
      });
      // move down a line
      expect(range2).toEqual([
        // line 1, item 0
        2,
        // line 1, item 1
        3,
        // line 2, item 0
        4,
        // line 2, item 1, focused
        5,
      ]);

      const range3 = updateVisibleRange({
        currentVisibleRange: range2,
        elementsPerLine,
        newFocusIndex: 0,
      });
      // move up a line
      expect(range3).toEqual([
        // line 0, item 0, focused
        0,
        // line 0, item 1
        1,
        // line 1, item 0
        2,
        // line 1, item 1
        3,
      ]);
    });
  });

  describe('with buffer lines', () => {
    const leadingBufferLines = 1;
    const trailingBufferLines = 1;

    it('works on straight line', () => {
      const elementsPerLine = 1;
      const range0 = [
        // leading buffer 0
        -1,
        // line 0, focused
        0,
        // line 1
        1,
        // line 2
        2,
        // line 3
        3,
        // trailing buffer 0
        4,
      ];
      const range1 = updateVisibleRange({
        currentVisibleRange: range0,
        elementsPerLine,
        leadingBufferLines,
        newFocusIndex: 2,
        trailingBufferLines,
      });
      // float within existing without changing
      expect(range1).toEqual([
        // leading buffer 0
        -1,
        // line 0
        0,
        // line 1
        1,
        // line 2, focused
        2,
        // line 3
        3,
        // trailing buffer 0
        4,
      ]);

      const range2 = updateVisibleRange({
        currentVisibleRange: range1,
        elementsPerLine,
        leadingBufferLines,
        newFocusIndex: 4,
        trailingBufferLines,
      });
      // move down a line
      expect(range2).toEqual([
        // leading buffer 0
        0,
        // line 1
        1,
        // line 2
        2,
        // line 3
        3,
        // line 4, focused
        4,
        // trailing buffer 0
        5,
      ]);

      const range3 = updateVisibleRange({
        currentVisibleRange: range2,
        elementsPerLine,
        leadingBufferLines,
        newFocusIndex: 0,
        trailingBufferLines,
      });
      // move up a line
      expect(range3).toEqual([
        // leading buffer 0
        -1,
        // line 0, focused
        0,
        // line 1
        1,
        // line 2
        2,
        // line 3
        3,
        // trailing buffer 0
        4,
      ]);
    });

    it('works on grid', () => {
      const elementsPerLine = 2;
      const range0 = [
        // leading buffer 0, item 0
        -2,
        // leading buffer 0, item 1
        -1,
        // line 0, item 0, focused
        0,
        // line 0, item 1
        1,
        // line 1, item 0
        2,
        // line 1, item 1
        3,
        // trailing buffer 0, item 0
        4,
        // trailing buffer 0, item 1
        5,
      ];
      const range1 = updateVisibleRange({
        currentVisibleRange: range0,
        elementsPerLine,
        leadingBufferLines,
        newFocusIndex: 2,
        trailingBufferLines,
      });
      // float within existing without changing
      expect(range1).toEqual([
        // leading buffer 0, item 0
        -2,
        // leading buffer 0, item 1
        -1,
        // line 0, item 0
        0,
        // line 0, item 1
        1,
        // line 1, item 0, focused
        2,
        // line 1, item 1
        3,
        // trailing buffer 0, item 0
        4,
        // trailing buffer 0, item 1
        5,
      ]);

      const range2 = updateVisibleRange({
        currentVisibleRange: range1,
        elementsPerLine,
        leadingBufferLines,
        newFocusIndex: 5,
        trailingBufferLines,
      });
      // move up a line
      expect(range2).toEqual([
        // leading buffer 0, item 0
        0,
        // leading buffer 0, item 1
        1,
        // line 1, item 0
        2,
        // line 1, item 1
        3,
        // line 2, item 0
        4,
        // line 2, item 1, focused
        5,
        // trailing buffer 0, item 0
        6,
        // trailing buffer 0, item 1
        7,
      ]);

      const range3 = updateVisibleRange({
        currentVisibleRange: range2,
        elementsPerLine,
        leadingBufferLines,
        newFocusIndex: 0,
        trailingBufferLines,
      });
      // move up a line
      expect(range3).toEqual([
        // leading buffer 0, item 0
        -2,
        // leading buffer 0, item 1
        -1,
        // line 0, item 0, focused
        0,
        // line 0, item 1
        1,
        // line 1, item 0
        2,
        // line 1, item 1
        3,
        // trailing buffer 0, item 0
        4,
        // trailing buffer 0, item 1
        5,
      ]);
    });
  });
});
