import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import { formatLiveViewersCount, formatTotalViewCount } from 'tachyon-more-ui';
import { formatVideoLength } from 'tachyon-utils';
import {
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Layout,
  WhiteSpace,
} from 'twitch-core-ui';

type InfoTextProps = {
  game: string;
  isLive: boolean;
  recordedAt: string | null;
  viewCount: number;
};

// istanbul ignore next: trivial
export const InfoText: FC<InfoTextProps> = ({
  game,
  isLive,
  recordedAt,
  viewCount,
}) => {
  const { formatMessage, formatPastRelativeDate } = useIntl();

  let liveFor = 0;
  let relativeDate = '';
  if (isLive) {
    if (recordedAt) {
      const recordedTime = new Date(recordedAt);
      const currentTime = new Date();
      liveFor = currentTime.getTime() - recordedTime.getTime();
      // convert from ms to seconds
      liveFor /= 1000;
      // round to avoid decimals in string
      liveFor = Math.floor(liveFor);
      // ensure number is positive
      liveFor = Math.max(0, liveFor);
    }
  } else {
    if (recordedAt) {
      relativeDate = formatPastRelativeDate(new Date(recordedAt));
      // Capitalize the first character for languages where it is appropriate
      relativeDate =
        relativeDate.charAt(0).toLocaleUpperCase() + relativeDate.substring(1);
    }
  }

  return (
    <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
      <CoreText color={Color.Alt2} fontSize={FontSize.Size8}>
        {game}
      </CoreText>
      {((isLive && liveFor) || (!isLive && relativeDate)) && (
        <>
          <CoreText
            color={Color.Alt2}
            fontSize={FontSize.Size8}
            whiteSpace={WhiteSpace.PreWrap}
          >
            {' | '}
          </CoreText>
          <CoreText color={Color.Alt2} fontSize={FontSize.Size8}>
            {isLive ? formatVideoLength(liveFor) : relativeDate}
          </CoreText>
        </>
      )}
      <CoreText
        color={Color.Alt2}
        fontSize={FontSize.Size8}
        whiteSpace={WhiteSpace.PreWrap}
      >
        {' | '}
      </CoreText>
      <CoreText
        color={isLive ? Color.Live : Color.Alt2}
        fontSize={FontSize.Size8}
      >
        {isLive
          ? formatLiveViewersCount(formatMessage, viewCount)
          : formatTotalViewCount(formatMessage, viewCount)}
      </CoreText>
    </Layout>
  );
};

InfoText.displayName = 'InfoText';
