import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { StreamType, squareSize } from 'tachyon-more-ui';
import { useFocus } from 'tachyon-tv-nav';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  Background,
  Color,
  CoreText,
  FlexDirection,
  FontSize,
  FontWeight,
  JustifyContent,
  Layout,
  TextType,
  ThemeRegion,
  WhiteSpace,
  ZIndex,
} from 'twitch-core-ui';
import { LightThemeMap } from 'twitch-core-ui-tokens';
import { RouteName, renderTachyonLink } from '../../../routing';
import { FocusableLink } from '../FocusableLink';
import { GradientAvatar } from '../GradientAvatar';
import { TruncatedTextFlexContainer } from '../TruncatedTextFlexContainer';
import { InfoText } from './InfoText';
import type { PlaybackInfo_channel } from './__generated__/PlaybackInfo_channel.graphql';

export type PlaybackInfoProps = {
  channel: PlaybackInfo_channel;
  focusIndex: number;
  gameDisplayName?: string | null;
  isLive: boolean;
  recordedAt?: string | null;
  title?: string | null;
  viewCount: number;
};

type ScDisappearingContainerProps = {
  $visible: boolean;
};

export const ScDisappearingContainer = styled.div<ScDisappearingContainerProps>`
  backface-visibility: hidden;
  display: flex;
  flex-direction: column;
  justify-content: center;
  ${({ $visible }) => !$visible && 'display: none;'}
  transform: translateZ(0);
`;

// isntanbul ignore next: trivial
const ScSpeechBubble = styled.div<ScDisappearingContainerProps>`
  background-color: ${LightThemeMap['color-background-alt']};
  border-radius: 0.5rem;
  opacity: ${({ $visible }) => ($visible ? 1 : 0)};
  padding: 1rem;

  &::after {
    border: 1.33rem solid transparent;
    border-left: 0;
    border-right-color: ${LightThemeMap['color-background-alt']};
    content: '';
    left: 0;
    ${squareSize({ sizePx: 0 })}
    position: absolute;
    top: 50%;
    transform: translate(-90%, -50%);
  }
`;

// isntanbul ignore next: trivial
export const PlaybackInfoBase: FC<PlaybackInfoProps> = ({
  channel,
  focusIndex,
  gameDisplayName,
  isLive,
  recordedAt,
  title,
  viewCount,
}) => {
  const { focused } = useFocus(focusIndex);
  const { formatMessage } = useIntl();
  const displayName = getFormattedUserDisplayName(channel);

  // TODO: determine why the ZIndex Above setting is required for the layout
  // wrapping the GradientAvatar. Without it, the image itself is visible, but
  // not the background gradient.
  return (
    <TruncatedTextFlexContainer>
      <Layout flexShrink={0} padding={{ right: 2 }} zIndex={ZIndex.Above}>
        <FocusableLink
          focusIndex={focusIndex}
          linkTo="/deferToRenderLink"
          renderLink={renderTachyonLink({
            route: RouteName.ChannelHome,
            routeParams: { login: channel.login },
          })}
        >
          <GradientAvatar
            login={channel.login}
            profileImageURL={channel.profileImageURL}
            streamType={isLive ? StreamType.Live : undefined}
            streamerPrimaryColorHex={channel.primaryColorHex}
          />
        </FocusableLink>
      </Layout>
      <TruncatedTextFlexContainer
        flexDirection={FlexDirection.Column}
        justifyContent={JustifyContent.Center}
      >
        <ScDisappearingContainer $visible={focused}>
          <ScSpeechBubble $visible={focused}>
            <ThemeRegion background={Background.Inherit} theme="light">
              <CoreText
                color={Color.Base}
                fontSize={FontSize.Size5}
                fontWeight={FontWeight.Bold}
                type={TextType.H3}
              >
                {displayName}
              </CoreText>
              <CoreText color={Color.Base} whiteSpace={WhiteSpace.NoWrap}>
                {formatMessage(
                  "Go to {displayName}'s profile...",
                  { displayName },
                  'BottomControlsStreamInfo',
                )}
              </CoreText>
            </ThemeRegion>
          </ScSpeechBubble>
        </ScDisappearingContainer>
        <ScDisappearingContainer $visible={!focused}>
          <CoreText
            ellipsis
            fontSize={FontSize.Size5}
            fontWeight={FontWeight.Bold}
            type={TextType.H3}
          >
            {displayName}
          </CoreText>
          {title && (
            <CoreText
              ellipsis
              fontSize={FontSize.Size5}
              fontWeight={FontWeight.Bold}
              type={TextType.H3}
            >
              {title}
            </CoreText>
          )}
          {gameDisplayName && (
            <InfoText
              game={gameDisplayName}
              isLive={isLive}
              recordedAt={recordedAt ?? null}
              viewCount={viewCount}
            />
          )}
        </ScDisappearingContainer>
      </TruncatedTextFlexContainer>
    </TruncatedTextFlexContainer>
  );
};

PlaybackInfoBase.displayName = 'PlaybackInfoBase';

export const PlaybackInfo = createFragmentContainer(PlaybackInfoBase, {
  channel: graphql`
    fragment PlaybackInfo_channel on User {
      displayName
      hosting {
        id
      }
      login
      profileImageURL(width: 70)
      primaryColorHex
    }
  `,
});

PlaybackInfo.displayName = 'PlaybackInfo';
