import type { PlaybackAd } from 'pulsar';
import type { FormatMessage } from 'tachyon-intl';
import { exhaustedCase, getFormattedUserDisplayName } from 'tachyon-utils';

export type AdChannelInfo = {
  displayName: string;
  login: string;
  roles: {
    isAffiliate: boolean | null;
    isPartner: boolean | null;
  } | null;
};

export type AdTextArgs = {
  ad: PlaybackAd;
  channel: AdChannelInfo | null;
  formatMessage: FormatMessage;
  live: boolean;
};

/**
 * The labels and logic in this file are approximate duplicates of Twilight's
 * implementation:
 * https://git.xarth.tv/twilight/twilight/blob/601ddc29db28f6e8b8129c3c406c3a99543e26ac/src/features/video-player/components/video-ads/video-ad-banner/utils.ts
 */

export function channelGetsAdRev({
  roles,
}: NonNullable<AdTextArgs['channel']>): boolean {
  return !!roles?.isAffiliate || !!roles?.isPartner;
}

// istanbul ignore next: trivial
export function defaultAdText(formatMessage: FormatMessage): string {
  return formatMessage('This Ad supports Twitch', 'defaultAdText');
}

// istanbul ignore next: trivial
function liveCatchAfterBreakText(
  channel: NonNullable<AdTextArgs['channel']>,
  formatMessage: FormatMessage,
): string {
  return formatMessage(
    'Catch {channel} right after this ad break; stick around to support the stream!',
    { channel: getFormattedUserDisplayName(channel) },
    'liveCatchAfterBreakText',
  );
}

// istanbul ignore next: trivial
export function preRollText({
  channel,
  formatMessage,
  live,
}: AdTextArgs): string {
  if (channel && channelGetsAdRev(channel)) {
    return live
      ? liveCatchAfterBreakText(channel, formatMessage)
      : formatMessage(
          'Start watching {channel} right after this ad break; stick around to support the channel!',
          { channel: getFormattedUserDisplayName(channel) },
          'preRollText',
        );
  }

  return defaultAdText(formatMessage);
}

// istanbul ignore next: trivial
export function midRollText({
  ad,
  channel,
  formatMessage,
  live,
}: AdTextArgs): string {
  if (live) {
    if (ad.vlm && channel) {
      return liveCatchAfterBreakText(channel, formatMessage);
    }

    if (!channel) {
      return formatMessage(
        'This stream will be back shortly. This Ad supports Twitch.',
        'midRollText',
      );
    }

    if (channelGetsAdRev(channel)) {
      return formatMessage(
        '{channel} is taking an ad break; stick around to support the stream!',
        { channel: getFormattedUserDisplayName(channel) },
        'midRollText',
      );
    }

    return formatMessage(
      '{channel} will be back shortly. This Ad supports Twitch.',
      { channel: getFormattedUserDisplayName(channel) },
      'midRollText',
    );
  }

  // VOD messaging
  return channel && channelGetsAdRev(channel)
    ? formatMessage(
        'Continue watching {channel} right after this ad break; stick around to support the channel!',
        { channel: getFormattedUserDisplayName(channel) },
        'midRollText',
      )
    : formatMessage(
        'This video will be back shortly. This Ad supports Twitch.',
        'midRollText',
      );
}

// istanbul ignore next: trivial
export function postRollText({
  channel,
  formatMessage,
  live,
}: AdTextArgs): string {
  if (live) {
    if (channel && channelGetsAdRev(channel)) {
      return formatMessage(
        'Thank you for watching {channel}; finish watching the ad to support the stream!',
        { channel: getFormattedUserDisplayName(channel) },
        'VideoAdBanner',
      );
    }

    return formatMessage(
      'Thanks for watching the stream! This Ad supports Twitch.',
      'VideoAdBanner',
    );
  }

  // Vod messaging
  if (channel && channelGetsAdRev(channel)) {
    return formatMessage(
      'Thank you for watching {channel}; finish watching the ad to support the channel!',
      { channel: getFormattedUserDisplayName(channel) },
      'VideoAdBanner',
    );
  }

  return defaultAdText(formatMessage);
}

// istanbul ignore next: trivial
export function getAdText(args: AdTextArgs): string {
  switch (args.ad.rollType) {
    case 'preroll':
      return preRollText(args);
    case 'midroll':
      return midRollText(args);
    case 'postroll':
      return postRollText(args);
    default:
      return exhaustedCase(args.ad.rollType, defaultAdText(args.formatMessage));
  }
}
