import type { PlaybackAd } from 'pulsar';
import type { FC } from 'react';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { formatVideoLength } from 'tachyon-utils';
import { CoreText, Display, Layout } from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import type { AdTextArgs } from './ad-text';
import { getAdText } from './ad-text';

export type { AdChannelInfo } from './ad-text';

const ScAdInfoContainer = styled.div`
  background: ${DarkThemeMap['color-background-modal-overlay']};
  display: flex;
  justify-content: space-between;
  left: 0;
  padding: 1rem;
  position: absolute;
  right: 0;
  top: 0;
`;

type PlayingAdInfoProps = Omit<AdTextArgs, 'ad' | 'formatMessage'> & {
  ad: PlaybackAd | null;
};

/**
 * Test channel: twitch.tv/adsbunny
 */
export const PlayingAdInfo: FC<PlayingAdInfoProps> = ({
  ad,
  channel,
  live,
}) => {
  const { formatMessage } = useIntl();
  if (!ad) {
    return null;
  }

  const { podCount, podPosition, remainingTimeSeconds } = ad;

  return (
    <ScAdInfoContainer>
      <CoreText>
        {getAdText({
          ad,
          channel,
          formatMessage,
          live,
        })}
      </CoreText>
      <Layout display={Display.Flex}>
        <CoreText>{formatVideoLength(remainingTimeSeconds)}</CoreText>
        {podCount > 1 && (
          <Layout padding={{ left: 2 }}>
            <CoreText>
              {formatMessage(
                '{adPosition} of {numberOfAds}',
                {
                  // podPosition starts at 0
                  adPosition: podPosition + 1,
                  numberOfAds: podCount,
                },
                'PlayingAdInfo',
              )}
            </CoreText>
          </Layout>
        )}
      </Layout>
    </ScAdInfoContainer>
  );
};

PlayingAdInfo.displayName = 'PlayingAdInfo';
