import {
  ClosedCaptionsRoot,
  hasCaptionsEnabledCookie,
  usePlaybackAd,
} from 'pulsar';
import type { FC } from 'react';
import { createContext, useContext, useEffect } from 'react';
import styled from 'styled-components';
import { useConst } from 'tachyon-utils';
import { Display, FlexDirection, Layout, Position } from 'twitch-core-ui';
import { OVERSCAN_PADDING_REM } from '../../../config';
import { useRecentUserInput } from '../../framework';
import { BackButton } from '../BackButton';
import { DEFAULT_HIDE_TRANSITION, HideableContainer } from '../styleMixins';
import { Captions } from './Captions';
import type { AdChannelInfo } from './PlayingAdInfo';
import { PlayingAdInfo } from './PlayingAdInfo';

// istanbul ignore next: trivial
const ScPlayerSection = styled.section`
  height: 100%;
  position: relative;
`;

// For padding-bottom: perfect 16:9 is 56.25%, but the video player content
// renders a tiny sliver higher than our overlay when chat is expanded so we
// over compensate by a little bit
const ScAdInfoOverlay = styled.div`
  bottom: 50%;
  left: 0;
  padding-bottom: 56.4%;
  position: absolute;
  right: 0;
  top: 50%;
  transform: translateY(-50%);
`;

// opacity is set to a tiny, but non-0, value to ensure it is read by
// screen readers (ChromeVox specifically).
// istanbul ignore next: trivial
const ScPlayerOverlay = styled.div<{ $darken: boolean }>`
  bottom: 0;
  left: 0;
  ${({ $darken }) =>
    $darken &&
    'background: linear-gradient(180deg, rgba(0, 0, 0, 0.4) 0%, #000000 88.43%, #000000 100%);'}
  position: absolute;
  right: 0;
  top: 0;
  transition: background ${DEFAULT_HIDE_TRANSITION};
`;

// We toggle between position absolute and just having padding to position
// the captions relative to the bottom controls when they're hidden.
// We align the captions left because it is easier to read with live content
// as the captions come in as incomplete sentences and get concatenated.
// istanbul ignore next: trivial
const ScCaptionsContainer = styled.div<{ $bottomControlsHidden: boolean }>`
  ${({ $bottomControlsHidden }) =>
    $bottomControlsHidden
      ? `position: absolute; bottom: ${OVERSCAN_PADDING_REM}rem;`
      : `padding-bottom: 2rem;`}
  width: fit-content;
`;

export const hidePlaybackUI = createContext(false);

export type PlayingUILayoutProps = {
  channel: AdChannelInfo | null;
  live: boolean;
  overlayChildren?: JSX.Element;
  player: JSX.Element;
};

/**
 * A component designed to make it easier to build a Playback UI.
 *
 * A typical overlay component might look something like:
 * <VerticalNav elementCount={...}>
 *   <PlayingUILayout player={<VideoPlayer />}>
 *     <PlayingUILayout.BackButton focusIndex={0} />
 *     <PlayingUILayout.Controls>
 *       {controls UI here}
 *     </PlayingUILayout.CTA>
 *   </PlayingUILayout>
 * </VerticalNav>
 */
export const PlayingUILayout: FC<PlayingUILayoutProps> & {
  BackButton: typeof BackButtonContainer;
  Controls: typeof Controls;
  TopRight: typeof TopRightContainer;
} = ({ channel, children, live, overlayChildren, player }) => {
  const { recentUserInput, simulateUserInput } = useRecentUserInput({
    defaultActive: true,
  });
  const ad = usePlaybackAd();

  // Hide the playing UI if an ad is running
  const showUI = !ad && recentUserInput;

  useEffect(() => {
    if (!ad) {
      return;
    }

    // only set this cleanup effect if it's the last pre-roll ad in the pod
    if (ad.rollType === 'preroll' && ad.podPosition + 1 === ad.podCount) {
      return () => {
        // Bring up the UI to match the default behavior of a channel that
        // doesn't encounter a pre-roll
        simulateUserInput();
      };
    }
  }, [ad, simulateUserInput]);

  return (
    <hidePlaybackUI.Provider value={!showUI}>
      <ScPlayerSection>
        {ad && (
          <ScAdInfoOverlay>
            <Layout
              attachLeft
              attachRight
              attachTop
              position={Position.Absolute}
            >
              <PlayingAdInfo ad={ad} channel={channel} live={live} />
            </Layout>
          </ScAdInfoOverlay>
        )}
        {player}
        <ScPlayerOverlay $darken={showUI}>{overlayChildren}</ScPlayerOverlay>
        {children}
      </ScPlayerSection>
    </hidePlaybackUI.Provider>
  );
};

// istanbul ignore next: trivial
const BackButtonContainer: FC<{ focusIndex: number }> = ({ focusIndex }) => {
  const hide = useContext(hidePlaybackUI);

  return (
    <Layout
      attachLeft
      attachTop
      padding={{ top: 3, x: OVERSCAN_PADDING_REM }}
      position={Position.Absolute}
    >
      <HideableContainer $hide={hide}>
        <BackButton focusIndex={focusIndex} withUnfocusBackground />
      </HideableContainer>
    </Layout>
  );
};

// istanbul ignore next: trivial
const Controls: FC = ({ children }) => {
  const captionsDefaultEnabled = useConst(hasCaptionsEnabledCookie);
  const hide = useContext(hidePlaybackUI);

  return (
    <ClosedCaptionsRoot
      contentKey="static"
      defaultEnabled={captionsDefaultEnabled}
    >
      <Layout
        attachBottom
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        fullWidth
        padding={{
          bottom: OVERSCAN_PADDING_REM,
          x: OVERSCAN_PADDING_REM,
        }}
        position={Position.Absolute}
      >
        <ScCaptionsContainer $bottomControlsHidden={hide}>
          <Captions />
        </ScCaptionsContainer>
        <HideableContainer $hide={hide}>{children}</HideableContainer>
      </Layout>
    </ClosedCaptionsRoot>
  );
};

const TopRightContainer: FC = ({ children }) => {
  const hide = useContext(hidePlaybackUI);

  return (
    <Layout
      attachRight
      attachTop
      padding={{ top: 3, x: OVERSCAN_PADDING_REM }}
      position={Position.Absolute}
    >
      <HideableContainer $hide={hide}>{children}</HideableContainer>
    </Layout>
  );
};

PlayingUILayout.BackButton = BackButtonContainer;
PlayingUILayout.Controls = Controls;
PlayingUILayout.TopRight = TopRightContainer;
BackButtonContainer.displayName = 'PlayingUILayout.BackButton';
Controls.displayName = 'PlayingUILayout.Controls';
PlayingUILayout.displayName = 'PlayingUILayout';
TopRightContainer.displayName = 'TopRightContainer';
