import {
  ClosedCaptionsRoot,
  hasCaptionsEnabledCookie,
  usePlaybackAd,
} from 'pulsar';
import type { FC } from 'react';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { recentUserInputContext } from '../../framework';
import { mockRecentUserInputContext } from '../../framework/InputActivity/RecentUserInput/test-mocks';
import { HideableContainer } from '../styleMixins';
import { PlayingAdInfo } from './PlayingAdInfo';
import type { PlayingUILayoutProps } from '.';
import { PlayingUILayout } from '.';

const MockConsumer: FC<Omit<PlayingUILayoutProps, 'player'>> = (props) => {
  return (
    <PlayingUILayout player={<div />} {...props}>
      <PlayingUILayout.BackButton focusIndex={0} />
      <PlayingUILayout.Controls children={<div />} />
    </PlayingUILayout>
  );
};

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  hasCaptionsEnabledCookie: jest.fn(),
  usePlaybackAd: jest.fn(),
}));
const mockHasCaptionsEnabledCookie = hasCaptionsEnabledCookie as jest.Mock;

describe(PlayingUILayout, () => {
  const setup = createMountWrapperFactory(
    MockConsumer,
    () => ({
      channel: {
        displayName: 'Therealderekt',
        login: 'therealderekt',
        roles: null,
      },
      live: true,
    }),
    {
      wrappingContexts: () => [
        mockRecentUserInputContext({
          recentUserInput: true,
        }),
      ],
    },
  );

  beforeEach(() => {
    (usePlaybackAd as jest.Mock).mockReturnValue(null);
  });

  describe('ads behavior', () => {
    it('hides the UI when recent input expires', () => {
      const { updateWrappingContext, wrapper } = setup();

      expect(wrapper.find(HideableContainer).first()).toHaveProp({
        $hide: false,
      });

      const [ctx, newVal] = mockRecentUserInputContext({
        recentUserInput: false,
      });
      updateWrappingContext(ctx, newVal);

      expect(wrapper.find(HideableContainer).first()).toHaveProp({
        $hide: true,
      });
    });

    it('shows the UI when no ad is playing', () => {
      const { wrapper } = setup();

      expect(wrapper.find(PlayingAdInfo)).not.toExist();
      expect(wrapper.find(HideableContainer).first()).toHaveProp({
        $hide: false,
      });
    });

    it('hides the normal UI while an ad is playing', () => {
      (usePlaybackAd as jest.Mock).mockReturnValue({ rollType: 'postroll' });
      const { wrapper } = setup();

      expect(wrapper.find(PlayingAdInfo)).toExist();
      expect(wrapper.find(HideableContainer).first()).toHaveProp({
        $hide: true,
      });
    });

    it('triggers the playing UI when a pre-roll ends', () => {
      (usePlaybackAd as jest.Mock).mockReturnValue({
        podCount: 2,
        podPosition: 1,
        rollType: 'preroll',
      });
      const { contexts, wrapper } = setup();

      // First call, is caused by the `defaultActive` flag
      expect(
        contexts.get(recentUserInputContext).simulateUserInput,
      ).toHaveBeenCalledTimes(1);

      (usePlaybackAd as jest.Mock).mockReturnValue(null);
      wrapper.setProps({});

      expect(
        contexts.get(recentUserInputContext).simulateUserInput,
      ).toHaveBeenCalledTimes(2);
    });

    it('does not trigger the playing UI when a non pre-roll ends', () => {
      (usePlaybackAd as jest.Mock).mockReturnValue({ rollType: 'midroll' });
      const { contexts, wrapper } = setup();
      // Just the initial defaultActive hook call
      expect(
        contexts.get(recentUserInputContext).simulateUserInput,
      ).toHaveBeenCalledTimes(1);

      (usePlaybackAd as jest.Mock).mockReturnValue(null);
      wrapper.setProps({});

      expect(
        contexts.get(recentUserInputContext).simulateUserInput,
      ).toHaveBeenCalledTimes(1);
    });
  });

  describe('captions', () => {
    it('defaults to showing captions when the cookie is set', () => {
      mockHasCaptionsEnabledCookie.mockReturnValue(true);
      const { wrapper } = setup();
      expect(wrapper.find(ClosedCaptionsRoot)).toHaveProp({
        defaultEnabled: true,
      });
    });

    it('defaults to hiding captions when the cookie is not set or false', () => {
      mockHasCaptionsEnabledCookie.mockReturnValue(false);
      const { wrapper } = setup();
      expect(wrapper.find(ClosedCaptionsRoot)).toHaveProp({
        defaultEnabled: false,
      });
    });
  });
});
