import type { FC, ReactNode } from 'react';
import styled from 'styled-components';
import { useStaticEnvironment } from 'tachyon-environment';
import type { QRUrlProps } from 'tachyon-more-ui';
import { QRUrl } from 'tachyon-more-ui';
import { CoreText, TextType } from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import {
  OVERSCAN_PADDING_REM,
  PRIMARY_QR_CODE_SIZE_REM,
  TOP_NAV_HEIGHT_REM,
} from '../../../config';

type QRLayoutProps = Pick<QRUrlProps, 'content' | 'url'>;

// istanbul ignore next: trivial
const BOX_PADDING_REM = 5;

// istanbul ignore next: trivial
const ScContent = styled.div`
  display: flex;
  flex-direction: row;
  padding-left: ${OVERSCAN_PADDING_REM}rem;
  padding-top: ${TOP_NAV_HEIGHT_REM}rem;
`;

// istanbul ignore next: trivial
const ScContentBoxLeft = styled.div`
  background: ${DarkThemeMap['color-background-base']};
  height: ${PRIMARY_QR_CODE_SIZE_REM + 4}rem;
  width: 60rem;
`;

// istanbul ignore next: trivial
const ScLeftPane = styled.div`
  display: flex;
  flex-direction: column;
  height: 100%;
  justify-content: center;
  padding: ${BOX_PADDING_REM}rem;
  padding-right: ${BOX_PADDING_REM * 2}rem;
  text-align: center;
`;

// istanbul ignore next: trivial
const ScQRUrlBox = styled.div`
  display: flex;
  flex-direction: column;
  justify-content: space-around;
  transform: translateX(-${BOX_PADDING_REM}rem);
`;

// istanbul ignore next: trivial
const ScQRPrompt = styled(CoreText)`
  padding: 2rem;
`;

// istanbul ignore next: trivial
const ScImageContainer = styled.div`
  height: 15rem;
  margin: 2rem;
  position: relative;
`;

/**
 * Component used to represent pages that contain a generic QR code.
 */
export const QRLayout: FC<QRLayoutProps> & {
  LeftPane: typeof LeftPane;
} = ({ children, content, url }) => {
  const {
    common: { platform },
  } = useStaticEnvironment();

  return (
    <ScContent>
      <ScContentBoxLeft>{children}</ScContentBoxLeft>
      <ScQRUrlBox>
        <QRUrl
          content={content}
          medium={`starshot_${platform}`}
          sizeRem={PRIMARY_QR_CODE_SIZE_REM}
          url={url}
        />
      </ScQRUrlBox>
    </ScContent>
  );
};

/**
 * Component used for consistent presentation of supplemental UI for a QR code.
 *
 * @example
 * <QRLayout { ...qrLayoutProps }>
 *   <QRLayout.LeftPane
 *      header={formatMessage('header message', 'pageName')}
 *      subText={formatMessage('sub text', 'pageName')}
 *      svg={<GlowExampleSvg />}
 *   />
 * </QRLayout>
 */
const LeftPane: FC<{ header: string; subText: string; svg?: ReactNode }> = ({
  header,
  subText,
  svg,
}) => (
  <ScLeftPane>
    {svg && <ScImageContainer>{svg}</ScImageContainer>}
    <CoreText type={TextType.H3}>{header}</CoreText>
    <ScQRPrompt type={TextType.H4}>{subText}</ScQRPrompt>
  </ScLeftPane>
);

QRLayout.displayName = 'QRLayoutComponent';
QRLayout.LeftPane = LeftPane;
LeftPane.displayName = 'QRLayoutComponent.LeftPane';
