import type { FC } from 'react';
import styled from 'styled-components';
import { RemSVG, ShadowedSVG } from 'tachyon-more-ui';
import {
  AlignItems,
  Background,
  BorderRadius,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  SVGAsset,
  TextType,
  Title,
  TitleSize,
} from 'twitch-core-ui';

const SVG_HEIGHT_REM = 4;

/**
 * Needs to be taller than the SVG within it, but otherwise arbitrarily
 * sized to what looks good.
 */
export const MENU_OPTION_HEIGHT_REM = SVG_HEIGHT_REM + 4;

const ScMenuOptionContainer = styled(Layout)`
  min-height: ${MENU_OPTION_HEIGHT_REM}rem;
`;

export type MenuOptionProps = {
  heading: string;
  /**
   * Sets an icon that visually represents the option.
   */
  icon?: SVGAsset;
  /**
   * Set to true if clicking the menu option leads to another menu.
   */
  subMenuIcon?: true;
  subheading?: JSX.Element | string;
};

// istanbul ignore next: trivial
export const MenuOption: FC<MenuOptionProps> = ({
  children,
  heading,
  icon,
  subMenuIcon,
  subheading,
}) => (
  <ScMenuOptionContainer
    alignItems={AlignItems.Center}
    background={Background.Alt2}
    borderRadius={BorderRadius.Large}
    color={Color.Base}
    display={Display.Flex}
    flexDirection={FlexDirection.Row}
    justifyContent={JustifyContent.Between}
    padding={{ x: 2, y: 1 }}
  >
    <Layout
      alignItems={AlignItems.Center}
      display={Display.Flex}
      flexDirection={FlexDirection.Row}
    >
      <Layout padding={{ right: 2 }}>
        {icon && (
          <ShadowedSVG
            asset={icon}
            heightRem={SVG_HEIGHT_REM}
            widthRem={SVG_HEIGHT_REM}
          />
        )}
      </Layout>
      <div>
        <Title size={TitleSize.Small} type={TextType.H2}>
          {heading}
        </Title>
        {subheading && (
          <Layout margin={{ top: 0.5 }}>
            <CoreText fontSize={FontSize.Size5}>{subheading}</CoreText>
          </Layout>
        )}
        {children && <Layout margin={{ top: 0.5 }}>{children}</Layout>}
      </div>
    </Layout>
    {subMenuIcon && (
      <RemSVG
        asset={SVGAsset.AngleRight}
        heightRem={SVG_HEIGHT_REM}
        widthRem={SVG_HEIGHT_REM}
      />
    )}
  </ScMenuOptionContainer>
);

MenuOption.displayName = 'MenuButton';
