import type { FC, ReactElement } from 'react';
import styled from 'styled-components';
import { VerticalNav } from 'tachyon-tv-nav';
import { Layout, Position, ZIndex } from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import {
  CARD_ANIMATION_TIMING_S,
  OVERSCAN_PADDING_REM,
  TOP_NAV_HEIGHT_REM,
} from '../../../config';
import type { MovableListProps } from '../MovableList';
import { MovableList } from '../MovableList';
import type { FocusableSettingsMenuOptionProps } from './FocusableSettingsMenuOption';
import {
  FOCUSABLE_SETTINGS_MENU_OPTION_HEIGHT_REM,
  FocusableSettingsMenuOption,
} from './FocusableSettingsMenuOption';

export * from './MenuOption';

// 1920px - (4 (overscan rem) x 18px x 2) = 1776 => 1776 / 1920 = 0.925
const SETTINGS_MENU_WIDTH_VW = 92.5;

const ScOutOfViewMenuOptionHider = styled.div`
  background-color: ${DarkThemeMap['color-background-body']};
  height: ${TOP_NAV_HEIGHT_REM + OVERSCAN_PADDING_REM}rem;
  z-index: 1;
`;

export type SettingsMenuOption = Omit<
  FocusableSettingsMenuOptionProps,
  'focusIndex'
>;

function settingsMenuItemRenderer(
  menuOption: SettingsMenuOption,
  focusIndex: number,
): ReactElement {
  return (
    <FocusableSettingsMenuOption
      focusIndex={focusIndex}
      key={menuOption.heading}
      {...menuOption}
    />
  );
}

type SettingsMenuProps = Pick<MovableListProps<SettingsMenuOption>, 'items'>;

// istanbul ignore next: trivial
/**
 * Provides a common layout for creating settings menu pages consisting of a title, optional
 * sub-heading, as well as efficiently rendering a list of focusable settings menu options.
 *
 * Use SettingsPrompt instead for settings menu pages that are confirmation pages or that perform other
 * actions that are not best represented as a list of options.
 *
 * Using Position.Fixed + z-index ensures that items moving into or out of view go behind the TopNav which looks
 * nicer.
 */
export const SettingsMenu: FC<SettingsMenuProps> = ({ items }) => (
  <Layout fullWidth position={Position.Fixed}>
    <ScOutOfViewMenuOptionHider />
    <Layout
      padding={{ top: 0.5, x: OVERSCAN_PADDING_REM }}
      position={Position.Relative}
      zIndex={ZIndex.Below}
    >
      <VerticalNav
        elementCount={items.length}
        focusIndex={0}
        handleWheel
        takeFocusOnFirstRender
      >
        <MovableList
          direction="vertical"
          elementsPerLine={1}
          itemRenderer={settingsMenuItemRenderer}
          items={items}
          leadingBufferLines={1}
          lineSizeRem={FOCUSABLE_SETTINGS_MENU_OPTION_HEIGHT_REM}
          trailingBufferLines={1}
          transitionDurationS={CARD_ANIMATION_TIMING_S}
          verticalWidthVw={SETTINGS_MENU_WIDTH_VW}
          visibleLines={5}
        />
      </VerticalNav>
    </Layout>
  </Layout>
);

SettingsMenu.displayName = 'SettingsMenu';
