import type { FC } from 'react';
import type { CSSObject } from 'styled-components';
import styled from 'styled-components';
import { useCurrentUser } from 'tachyon-auth';
import { HorizontalNav } from 'tachyon-tv-nav';
import {
  AlignItems,
  Display,
  JustifyContent,
  Layout,
  Position,
  ZIndex,
} from 'twitch-core-ui';
import { DarkThemeMap } from 'twitch-core-ui-tokens';
import { OVERSCAN_PADDING_REM, TOP_NAV_HEIGHT_REM } from '../../../config';
import { BackButton } from '../BackButton';
import { Glitch } from './Glitch';
import { NavIcons } from './NavIcons';
import { NavMenu } from './NavMenu';

type ScTopNavContainerProps = {
  /**
   * Controls whether the TopNav renders with a background
   */
  $withBackground?: true;
};

function getBackground({ $withBackground }: ScTopNavContainerProps): CSSObject {
  if (!$withBackground) {
    return {};
  }

  return {
    background: DarkThemeMap['color-background-body'],
  };
}

export type TopNavDisplayOptions = {
  /**
   * Determines whether the "Go Back" button is shown.
   */
  displayBackButton?: true;
  /**
   * Communicates whether the page will render with a TopNav. This allows the
   * loading component to match the next page when transitioning between pages.
   */
  displayNavMenu?: true;
};

const ScTopNavContainer = styled(Layout)<ScTopNavContainerProps>`
  height: ${TOP_NAV_HEIGHT_REM}rem;
  ${getBackground}
  width: 100vw;
`;

type TopNavProps = TopNavDisplayOptions & {
  focusIndex: number;
  withBackground?: true;
};

export const TopNav: FC<TopNavProps> = (props) => {
  const { displayBackButton, displayNavMenu, focusIndex, withBackground } =
    props;
  const { loggedIn } = useCurrentUser();

  const elementCount = (displayBackButton ? 1 : 0) + (displayNavMenu ? 2 : 0);

  /**
   * Take note that the left container (Glitch or BackButton) is anchored left.
   * The right container (NavIcons) is anchored right.
   * And the central container (NavMenu) is absolutely centered.
   * This is to prevent the dynamic size of the left component (Glitch/Backbutton)
   * from screwing up the centering of the NavMenu.
   */

  return (
    <HorizontalNav elementCount={elementCount} focusIndex={focusIndex}>
      <ScTopNavContainer
        $withBackground={withBackground}
        alignItems={AlignItems.Center}
        as="nav"
        attachLeft
        attachRight
        attachTop
        position={Position.Absolute}
        zIndex={ZIndex.Above}
      >
        <Layout
          alignItems={AlignItems.Center}
          attachBottom
          attachLeft
          attachRight
          attachTop
          display={Display.Flex}
          margin={{
            top: OVERSCAN_PADDING_REM,
            x: OVERSCAN_PADDING_REM,
          }}
          position={Position.Absolute}
        >
          <Layout
            attachLeft
            padding={{ left: OVERSCAN_PADDING_REM }}
            position={Position.Relative}
            zIndex={ZIndex.Above}
          >
            {displayBackButton ? <BackButton focusIndex={0} /> : <Glitch />}
          </Layout>
          {displayNavMenu && (
            <>
              <Layout
                alignItems={AlignItems.Center}
                attachBottom
                attachLeft
                attachRight
                attachTop
                display={Display.Flex}
                justifyContent={JustifyContent.Center}
                position={Position.Absolute}
              >
                <NavMenu
                  focusIndex={displayBackButton ? 1 : 0}
                  loggedIn={loggedIn}
                />
              </Layout>
              <Layout
                attachRight
                margin={{ left: 'auto' }}
                padding={{ right: OVERSCAN_PADDING_REM }}
                position={Position.Relative}
              >
                <NavIcons focusIndex={displayBackButton ? 2 : 1} />
              </Layout>
            </>
          )}
        </Layout>
      </ScTopNavContainer>
    </HorizontalNav>
  );
};

TopNav.displayName = 'TopNav';
