import type { FC } from 'react';
import styled from 'styled-components';
import { BorderRadius, ProgressBar, ProgressBarSize } from 'twitch-core-ui';
import { DarkThemeMap, StaticTokenMap } from 'twitch-core-ui-tokens';

export type ProgressBarOverlayProps = {
  duration: number | null;
  position: number | null;
};

export const clamp = (val: number, lower: number, upper: number): number => {
  const upperClamp = Math.min(val, upper);
  return Math.max(lower, upperClamp);
};

// Either computes a value from 0 to 100 representing the playbaack percentage
// or returns undefined if that value can't be computed or if playback is at position 0
const calculateProgressPercent = (
  duration: number,
  position: number,
): number => {
  return clamp(100 * (position / duration), 0, 100);
};

const ScProgressBarContainer = styled.div`
  background: ${StaticTokenMap['color-black']};
  color: ${DarkThemeMap['color-background-accent']};
`;

export const ProgressBarOverlay: FC<ProgressBarOverlayProps> = ({
  duration,
  position,
}) => {
  // These checks are just to be extra defensive in case we get some bad data from GQL
  if (!duration || !position || duration <= 0 || position <= 0) {
    return null;
  }

  return (
    <ScProgressBarContainer>
      <ProgressBar
        borderRadius={BorderRadius.None}
        inherit
        mask
        overlay={false}
        size={ProgressBarSize.Small}
        value={calculateProgressPercent(duration, position)}
      />
    </ScProgressBarContainer>
  );
};
ProgressBarOverlay.displayName = 'ProgressBarOverlay';
