import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  formatTotalViewCount,
  getBoxArtSrcAndSrcSet,
  getOverlaySrcAndSrcSet,
} from 'tachyon-more-ui';
import { formatVideoLength } from 'tachyon-utils';
import {
  AspectRatio,
  CardImage,
  Color,
  CoreImage,
  CoreText,
  Layout,
  MediaCard,
  MediaCardImage,
  MediaCardMeta,
  MediaCardStat,
  Position,
} from 'twitch-core-ui';
import {
  VIDEO_CARD_BOX_ART_SIZE_REM,
  WATCHABLE_CARD_CONFIG,
} from '../../../config';
import { FocusableCardImageContainer } from '../FocusableCardImageContainer';
import { ProgressBarOverlay } from './ProgressBarOverlay';

// Convenience type for representing data associated with a video. Clips and VODs
// have a slightly different shape coming from GraphQL, but this forces them
// into a standard metadata object.
type VideoCardMeta = {
  createdAt: string | null;
  game: {
    boxArtURL: string | null;
    displayName: string;
  } | null;
  lengthSeconds: number | null;
  previewThumbnailURL: string;
  self: {
    viewingHistory: {
      position: number | null;
    } | null;
  } | null;
  title: string | null;
  viewCount: number | null;
};

type VideoCardProps = {
  focusIndex: number;
  video: VideoCardMeta;
};

// istanbul ignore next: trivial
// Ensure that props provided to this component remain stable to avoid expensive
// re-renders. Also avoid using context-hooks directly in here for the same
// reason; instead push them into child components like FocusableCardImageContainer.
export const VideoCard: FC<VideoCardProps> = ({ focusIndex, video }) => {
  const { formatMessage, formatNumberShort, formatPastRelativeDate } =
    useIntl();

  let createdAt = '';
  if (video.createdAt) {
    createdAt = formatPastRelativeDate(new Date(video.createdAt));
    createdAt =
      createdAt.charAt(0).toLocaleUpperCase() + createdAt.substring(1);
  }

  return (
    <MediaCard
      image={
        <FocusableCardImageContainer
          config={WATCHABLE_CARD_CONFIG}
          focusIndex={focusIndex}
        >
          <MediaCardImage
            bottomLeft={
              video.viewCount ? (
                <MediaCardStat>
                  {formatTotalViewCount(
                    formatMessage,
                    video.viewCount,
                    formatNumberShort(video.viewCount, {
                      maximumSignificantDigits: 2,
                    }),
                  )}
                </MediaCardStat>
              ) : null
            }
            bottomRight={
              video.lengthSeconds ? (
                <MediaCardStat>
                  {formatVideoLength(video.lengthSeconds)}
                </MediaCardStat>
              ) : null
            }
            image={
              <div>
                <CardImage
                  alt=""
                  aspect={AspectRatio.Aspect16x9}
                  sizes={[{ size: `${WATCHABLE_CARD_CONFIG.widthRem}rem` }]}
                  {...getOverlaySrcAndSrcSet(video.previewThumbnailURL)}
                />
                <Layout attachBottom fullWidth position={Position.Absolute}>
                  <ProgressBarOverlay
                    duration={video.lengthSeconds}
                    position={video.self?.viewingHistory?.position ?? null}
                  />
                </Layout>
              </div>
            }
            ratio={AspectRatio.Aspect16x9}
          />
        </FocusableCardImageContainer>
      }
      meta={
        <MediaCardMeta
          image={
            video.game?.boxArtURL && (
              <CoreImage
                alt=""
                sizes={[{ size: `${VIDEO_CARD_BOX_ART_SIZE_REM}rem` }]}
                {...getBoxArtSrcAndSrcSet(video.game.boxArtURL)}
              />
            )
          }
          links={
            <>
              {video.createdAt ? (
                <CoreText children={createdAt} color={Color.Base} />
              ) : null}
              {video.game?.displayName ? (
                <CoreText
                  children={video.game.displayName}
                  color={Color.Base}
                  ellipsis
                />
              ) : null}
            </>
          }
          title={video.title ?? ''}
        />
      }
    />
  );
};

VideoCard.displayName = 'VideoCard';
